'use server';

/**
 * @fileOverview AI-powered intelligent product search flow.
 *
 * - intelligentProductSearch - A function that handles the intelligent product search process.
 * - IntelligentProductSearchInput - The input type for the intelligentProductSearch function.
 * - IntelligentProductSearchOutput - The return type for the intelligentProductSearch function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const IntelligentProductSearchInputSchema = z.object({
  query: z.string().describe('The user search query, which can be in natural language.'),
});
export type IntelligentProductSearchInput = z.infer<
  typeof IntelligentProductSearchInputSchema
>;

const IntelligentProductSearchOutputSchema = z.object({
  suggestedProducts: z
    .array(z.string())
    .describe(
      'A list of suggested product names based on the user query, considering medical conditions and related items.'
    ),
});
export type IntelligentProductSearchOutput = z.infer<
  typeof IntelligentProductSearchOutputSchema
>;

export async function intelligentProductSearch(
  input: IntelligentProductSearchInput
): Promise<IntelligentProductSearchOutput> {
  return intelligentProductSearchFlow(input);
}

const productSearchTool = ai.defineTool(
  {
    name: 'getProductSuggestions',
    description:
      'Suggests relevant products based on a user query, considering medical conditions and related items.',
    inputSchema: z.object({
      query: z
        .string()
        .describe(
          'The user search query, which can be in natural language or related terms.'
        ),
    }),
    outputSchema: z.array(z.string()),
  },
  async (input) => {
    // Mock implementation - replace with actual product search logic
    const products = [
      'Product A',
      'Product B',
      'Product C',
      'Medical Device X',
      'Supplement Y',
    ];
    return products.filter(product =>
      product.toLowerCase().includes(input.query.toLowerCase())
    );
  }
);

const prompt = ai.definePrompt({
  name: 'intelligentProductSearchPrompt',
  input: {schema: IntelligentProductSearchInputSchema},
  output: {schema: IntelligentProductSearchOutputSchema},
  tools: [productSearchTool],
  prompt: `Based on the user query: {{{query}}}, suggest a list of relevant products, considering medical conditions and related items. Use the getProductSuggestions tool to find products.

  Return the list of suggested products in the following format:
  {
    "suggestedProducts": ["Product 1", "Product 2", ...]
  }`,
});

const intelligentProductSearchFlow = ai.defineFlow(
  {
    name: 'intelligentProductSearchFlow',
    inputSchema: IntelligentProductSearchInputSchema,
    outputSchema: IntelligentProductSearchOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
