
'use server';

/**
 * @fileOverview AI-powered live chat agent for customer support.
 *
 * - liveChat - A function that provides automated responses to user queries.
 * - LiveChatInput - The input type for the liveChat function.
 * - LiveChatOutput - The return type for the liveChat function.
 */

import { ai } from '@/ai/genkit';
import { z } from 'genkit';

const LiveChatInputSchema = z.object({
  query: z.string().describe('The user\'s question or problem.'),
  category: z.string().describe('The support category selected by the user (e.g., "Order Issues", "Payment & Refunds").'),
});
export type LiveChatInput = z.infer<typeof LiveChatInputSchema>;

const LiveChatOutputSchema = z.object({
  response: z.string().describe('A helpful and conversational response to the user\'s query.'),
});
export type LiveChatOutput = z.infer<typeof LiveChatOutputSchema>;

export async function liveChat(input: LiveChatInput): Promise<LiveChatOutput> {
  return liveChatFlow(input);
}

const prompt = ai.definePrompt({
  name: 'liveChatPrompt',
  input: { schema: LiveChatInputSchema },
  output: { schema: LiveChatOutputSchema },
  prompt: `You are a friendly and helpful customer support agent for ChandraBrothers, an online pharmacy.
  Your goal is to assist users with their questions based on the category they've selected.
  Keep your answers concise, clear, and empathetic.

  You must never give medical advice. If a user asks for medical advice, gently decline and advise them to consult a healthcare professional or use the "Pharmacy Assistance" contact option.

  User's selected category: {{{category}}}
  User's question: {{{query}}}

  Provide a helpful response.`,
});

const liveChatFlow = ai.defineFlow(
  {
    name: 'liveChatFlow',
    inputSchema: LiveChatInputSchema,
    outputSchema: LiveChatOutputSchema,
  },
  async (input) => {
    const { output } = await prompt(input);
    return output!;
  }
);
