
'use client';

import { useState } from 'react';
import * as LucideIcons from 'lucide-react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from '@/components/ui/alert-dialog';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
} from '@/components/ui/dialog';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  PlusCircle,
  LayoutList,
  MoreHorizontal,
  Edit,
  Trash2,
  Package,
  LoaderCircle,
  AlertTriangle,
} from 'lucide-react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection } from 'firebase/firestore';
import type { CategoryData } from '@/lib/data';
import { CategoryForm } from '@/components/category-form';
import { Skeleton } from '@/components/ui/skeleton';
import { useToast } from '@/hooks/use-toast';
import { deleteCategory } from '@/firebase/firestore/categories';

// A type guard to check if a string is a valid Lucide icon name
function isLucideIcon(name: string): name is keyof typeof LucideIcons {
  return name in LucideIcons;
}

function CategoryIcon({ name }: { name: string }) {
  if (isLucideIcon(name)) {
    const Icon = LucideIcons[name];
    return <Icon className="h-5 w-5 text-muted-foreground" />;
  }
  // Fallback icon if the name is invalid
  return <Package className="h-5 w-5 text-muted-foreground" />;
}

export default function AdminCategoriesPage() {
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState<CategoryData | undefined>(undefined);
  const [isDeleting, setIsDeleting] = useState(false);
  const firestore = useFirestore();
  const { toast } = useToast();

  const categoriesCollectionRef = useMemoFirebase(
    () => collection(firestore, 'categories'),
    [firestore]
  );
  const {
    data: categories,
    isLoading,
    error,
  } = useCollection<CategoryData>(categoriesCollectionRef);

  const handleEdit = (category: CategoryData) => {
    setSelectedCategory(category);
    setIsModalOpen(true);
  };

  const handleAddNew = () => {
    setSelectedCategory(undefined);
    setIsModalOpen(true);
  };
  
  const handleDelete = async (categoryId: string) => {
    setIsDeleting(true);
    try {
        await deleteCategory(firestore, categoryId);
        toast({
            title: 'Category Deleted',
            description: 'The category has been successfully removed.'
        })
    } catch (e: any) {
        toast({
            variant: 'destructive',
            title: 'Deletion Failed',
            description: 'Could not delete the category. Please try again.'
        })
    } finally {
        setIsDeleting(false);
    }
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <Dialog
        open={isModalOpen}
        onOpenChange={(isOpen) => {
          setIsModalOpen(isOpen);
          if (!isOpen) setSelectedCategory(undefined);
        }}
      >
        <div className="flex flex-col md:flex-row justify-between md:items-center gap-4">
          <div>
            <h1 className="text-3xl font-bold">Category Management</h1>
            <p className="text-muted-foreground">
              Add, edit, and organize your product categories.
            </p>
          </div>
          <DialogTrigger asChild>
            <Button onClick={handleAddNew}>
              <PlusCircle className="mr-2" />
              Add New Category
            </Button>
          </DialogTrigger>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>All Categories</CardTitle>
            <CardDescription>
              A list of all product categories in your store.
            </CardDescription>
          </CardHeader>
          <CardContent>
            {isLoading && (
              <div className="space-y-2">
                {[...Array(3)].map((_, i) => (
                  <Skeleton key={i} className="h-12 w-full" />
                ))}
              </div>
            )}
            {error && (
              <Alert variant="destructive">
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle>Error</AlertTitle>
                <AlertDescription>
                  Failed to load categories. Please try again.
                </AlertDescription>
              </Alert>
            )}
            {!isLoading && !error && categories && categories.length > 0 && (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Icon</TableHead>
                    <TableHead>Name</TableHead>
                    <TableHead>Banner URL</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {categories.map((category) => (
                    <TableRow key={category.id}>
                      <TableCell>
                        <CategoryIcon name={category.icon} />
                      </TableCell>
                      <TableCell className="font-medium">{category.name}</TableCell>
                      <TableCell className="text-muted-foreground truncate max-w-xs">
                        {category.bannerUrl || 'Not set'}
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" size="icon">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem onClick={() => handleEdit(category)}>
                              <Edit className="mr-2 h-4 w-4" />
                              Edit
                            </DropdownMenuItem>
                            <AlertDialog>
                              <AlertDialogTrigger asChild>
                                <DropdownMenuItem onSelect={(e) => e.preventDefault()} className="text-destructive">
                                  <Trash2 className="mr-2 h-4 w-4" />
                                  Delete
                                </DropdownMenuItem>
                              </AlertDialogTrigger>
                              <AlertDialogContent>
                                <AlertDialogHeader>
                                  <AlertDialogTitle>Are you sure?</AlertDialogTitle>
                                  <AlertDialogDescription>
                                    This action cannot be undone. This will permanently delete the "{category.name}" category.
                                  </AlertDialogDescription>
                                </AlertDialogHeader>
                                <AlertDialogFooter>
                                  <AlertDialogCancel>Cancel</AlertDialogCancel>
                                  <AlertDialogAction onClick={() => handleDelete(category.id!)} disabled={isDeleting} className="bg-destructive hover:bg-destructive/90">
                                     {isDeleting && <LoaderCircle className="animate-spin mr-2 h-4 w-4" />}
                                    Delete
                                  </AlertDialogAction>
                                </AlertDialogFooter>
                              </AlertDialogContent>
                            </AlertDialog>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            )}
            {!isLoading && !error && (!categories || categories.length === 0) && (
              <Alert>
                <LayoutList className="h-4 w-4" />
                <AlertTitle>No Categories Found</AlertTitle>
                <AlertDescription>
                  There are no categories to display. Click "Add New Category"
                  to get started.
                </AlertDescription>
              </Alert>
            )}
          </CardContent>
        </Card>

        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {selectedCategory ? 'Edit Category' : 'Add New Category'}
            </DialogTitle>
             <DialogDescription>
                {selectedCategory ? `Editing the "${selectedCategory.name}" category.` : 'Fill in the details to create a new category.'}
            </DialogDescription>
          </DialogHeader>
          <CategoryForm
            category={selectedCategory}
            onSave={() => setIsModalOpen(false)}
          />
        </DialogContent>
      </Dialog>
    </div>
  );
}

    