'use client';

import { useState } from 'react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Button } from '@/components/ui/button';
import {
  Palette,
  PlusCircle,
  Image,
  AlertTriangle,
  LoaderCircle,
  MoreHorizontal,
  Edit,
  Trash2,
} from 'lucide-react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, deleteDoc, doc } from 'firebase/firestore';
import type { HeroContent } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { Switch } from '@/components/ui/switch';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
} from '@/components/ui/dialog';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from '@/components/ui/alert-dialog';
import { useToast } from '@/hooks/use-toast';
import { HeroContentForm } from '@/components/hero-content-form';

export default function AdminContentPage() {
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedSlide, setSelectedSlide] = useState<HeroContent | undefined>(undefined);
  const [isDeleting, setIsDeleting] = useState<string | null>(null);
  const firestore = useFirestore();
  const { toast } = useToast();

  const heroContentCollectionRef = useMemoFirebase(
    () => collection(firestore, 'hero_content'),
    [firestore]
  );
  const {
    data: slides,
    isLoading,
    error,
  } = useCollection<HeroContent>(heroContentCollectionRef);

  const handleAddNew = () => {
    setSelectedSlide(undefined);
    setIsModalOpen(true);
  };
  
  const handleEdit = (slide: HeroContent) => {
    setSelectedSlide(slide);
    setIsModalOpen(true);
  }

  const handleDelete = async (slideId: string) => {
    setIsDeleting(slideId);
    try {
        await deleteDoc(doc(firestore, 'hero_content', slideId));
        toast({
            title: 'Slide Deleted',
            description: 'The hero slide has been removed.',
        });
    } catch(e) {
        toast({
            variant: 'destructive',
            title: 'Deletion Failed',
            description: 'Could not delete the slide.',
        });
    } finally {
        setIsDeleting(null);
    }
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <Dialog
        open={isModalOpen}
        onOpenChange={(isOpen) => {
          setIsModalOpen(isOpen);
          if (!isOpen) setSelectedSlide(undefined);
        }}
      >
        <div className="flex flex-col md:flex-row justify-between md:items-center gap-4">
          <div>
            <h1 className="text-3xl font-bold">Content Management</h1>
            <p className="text-muted-foreground">
              Manage banners, sliders, and other page content.
            </p>
          </div>
           <DialogTrigger asChild>
            <Button onClick={handleAddNew}>
              <PlusCircle className="mr-2" />
              Add New Slide
            </Button>
          </DialogTrigger>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
                <Image className="h-5 w-5" />
                Hero Slider Content
            </CardTitle>
            <CardDescription>
              Manage the slides that appear in the main homepage carousel.
            </CardDescription>
          </CardHeader>
          <CardContent>
            {isLoading && (
              <div className="space-y-2">
                {[...Array(3)].map((_, i) => (
                  <Skeleton key={i} className="h-20 w-full" />
                ))}
              </div>
            )}
            {error && (
              <Alert variant="destructive">
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle>Error Loading Content</AlertTitle>
                <AlertDescription>
                  Could not fetch hero content. Please check your permissions.
                </AlertDescription>
              </Alert>
            )}
            {!isLoading && !error && slides && slides.length > 0 && (
              <div className="divide-y">
                {slides.map((slide) => (
                  <div key={slide.id} className="flex items-center justify-between py-3">
                    <div className="flex-1">
                        <p className="font-semibold">{slide.headline}</p>
                        <p className="text-sm text-muted-foreground truncate">{slide.subheading}</p>
                    </div>
                    <div className="flex items-center gap-4 ml-4">
                        <Switch checked={slide.isActive} />
                        <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                                <Button variant="ghost" size="icon" disabled={isDeleting === slide.id}>
                                    {isDeleting === slide.id ? <LoaderCircle className="h-4 w-4 animate-spin" /> : <MoreHorizontal className="h-4 w-4" />}
                                </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent>
                                <DropdownMenuItem onClick={() => handleEdit(slide)}>
                                    <Edit className="mr-2 h-4 w-4" /> Edit
                                </DropdownMenuItem>
                                <AlertDialog>
                                    <AlertDialogTrigger asChild>
                                        <DropdownMenuItem onSelect={e => e.preventDefault()} className="text-destructive">
                                            <Trash2 className="mr-2 h-4 w-4" /> Delete
                                        </DropdownMenuItem>
                                    </AlertDialogTrigger>
                                    <AlertDialogContent>
                                        <AlertDialogHeader>
                                            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
                                            <AlertDialogDescription>This will permanently delete the slide "{slide.headline}".</AlertDialogDescription>
                                        </AlertDialogHeader>
                                        <AlertDialogFooter>
                                            <AlertDialogCancel>Cancel</AlertDialogCancel>
                                            <AlertDialogAction onClick={() => handleDelete(slide.id!)} className="bg-destructive">Delete</AlertDialogAction>
                                        </AlertDialogFooter>
                                    </AlertDialogContent>
                                </AlertDialog>
                            </DropdownMenuContent>
                        </DropdownMenu>
                    </div>
                  </div>
                ))}
              </div>
            )}
             {!isLoading && !error && (!slides || slides.length === 0) && (
                 <Alert>
                    <Image className="h-4 w-4" />
                    <AlertTitle>No Slides Found</AlertTitle>
                    <AlertDescription>
                    There are no hero slides to display. Click "Add New Slide" to create one.
                    </AlertDescription>
                </Alert>
             )}
          </CardContent>
        </Card>

         <DialogContent className="sm:max-w-[700px]">
          <DialogHeader>
            <DialogTitle>
              {selectedSlide ? 'Edit Hero Slide' : 'Add New Hero Slide'}
            </DialogTitle>
            <DialogDescription>
              {selectedSlide ? `Editing the slide: "${selectedSlide.headline}"` : 'Fill in the details for the new slide.'}
            </DialogDescription>
          </DialogHeader>
          <HeroContentForm
            slide={selectedSlide}
            onSave={() => setIsModalOpen(false)}
          />
        </DialogContent>
      </Dialog>
    </div>
  );
}
