'use client';

import { useState } from 'react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
} from '@/components/ui/dialog';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
  PlusCircle,
  Percent,
  MoreHorizontal,
  Edit,
  Trash2,
  AlertTriangle,
  BarChart2,
} from 'lucide-react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection } from 'firebase/firestore';
import type { Coupon } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { useToast } from '@/hooks/use-toast';
import { CouponForm } from '@/components/coupon-form';
import { Badge } from '@/components/ui/badge';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  ResponsiveContainer,
} from 'recharts';

const couponUsageData = [
  { name: 'SUMMER20', usage: 150 },
  { name: 'NEWUSER10', usage: 300 },
  { name: 'SALE50', usage: 75 },
  { name: 'FREESHIP', usage: 400 },
];


export default function AdminCouponsPage() {
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedCoupon, setSelectedCoupon] = useState<Coupon | undefined>(undefined);
  const firestore = useFirestore();
  const { toast } = useToast();

  const couponsCollectionRef = useMemoFirebase(
    () => collection(firestore, 'coupons'),
    [firestore]
  );
  const {
    data: coupons,
    isLoading,
    error,
  } = useCollection<Coupon>(couponsCollectionRef);
  
  const handleEdit = (coupon: Coupon) => {
    setSelectedCoupon(coupon);
    setIsModalOpen(true);
  };

  const handleAddNew = () => {
    setSelectedCoupon(undefined);
    setIsModalOpen(true);
  };

  const getStatus = (expiryDate: any) => {
    if (!expiryDate?.seconds) return { text: 'No Expiry', variant: 'secondary' as const };
    const isExpired = new Date(expiryDate.seconds * 1000) < new Date();
    return isExpired
      ? { text: 'Expired', variant: 'destructive' as const }
      : { text: 'Active', variant: 'default' as const };
  }


  return (
    <div className="p-4 md:p-8 space-y-6">
       <Dialog
        open={isModalOpen}
        onOpenChange={(isOpen) => {
          setIsModalOpen(isOpen);
          if (!isOpen) setSelectedCoupon(undefined);
        }}
      >
        <div className="flex flex-col md:flex-row justify-between md:items-center gap-4">
            <div>
            <h1 className="text-3xl font-bold">Coupon Management</h1>
            <p className="text-muted-foreground">
                Create and manage discount coupons for your store.
            </p>
            </div>
            <DialogTrigger asChild>
                <Button onClick={handleAddNew}>
                <PlusCircle className="mr-2" />
                Create New Coupon
                </Button>
            </DialogTrigger>
        </div>
        
        <Card>
            <CardHeader>
                <CardTitle className="flex items-center gap-2">
                    <BarChart2 />
                    Coupon Usage Analytics
                </CardTitle>
                <CardDescription>A brief overview of how your coupons are performing.</CardDescription>
            </CardHeader>
            <CardContent>
                <ResponsiveContainer width="100%" height={250}>
                    <BarChart data={couponUsageData}>
                        <CartesianGrid strokeDasharray="3 3" />
                        <XAxis dataKey="name" fontSize={12} />
                        <YAxis fontSize={12} />
                        <Tooltip />
                        <Legend />
                        <Bar dataKey="usage" fill="hsl(var(--primary))" name="Times Used" />
                    </BarChart>
                </ResponsiveContainer>
            </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>All Coupons</CardTitle>
            <CardDescription>
              A list of all coupons available in your store.
            </CardDescription>
          </CardHeader>
          <CardContent>
            {isLoading && (
              <div className="space-y-2">
                {[...Array(3)].map((_, i) => (
                  <Skeleton key={i} className="h-12 w-full" />
                ))}
              </div>
            )}
            {error && (
              <Alert variant="destructive">
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle>Error</AlertTitle>
                <AlertDescription>
                  Failed to load coupons. Please ensure you have admin permissions.
                </AlertDescription>
              </Alert>
            )}
            {!isLoading && !error && coupons && coupons.length > 0 && (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Code</TableHead>
                    <TableHead>Discount</TableHead>
                    <TableHead>Min. Order</TableHead>
                    <TableHead>Expiry</TableHead>
                    <TableHead>Usage Limit</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {coupons.map((coupon) => {
                    const status = getStatus(coupon.expiryDate);
                    return (
                        <TableRow key={coupon.id}>
                            <TableCell className="font-mono font-bold">{coupon.code}</TableCell>
                            <TableCell>{coupon.discountPercentage}%</TableCell>
                            <TableCell>₹{coupon.minOrderAmount.toFixed(2)}</TableCell>
                            <TableCell>{coupon.expiryDate ? new Date(coupon.expiryDate.seconds * 1000).toLocaleDateString() : 'N/A'}</TableCell>
                            <TableCell>{coupon.maxUsageLimit}</TableCell>
                            <TableCell><Badge variant={status.variant}>{status.text}</Badge></TableCell>
                            <TableCell className="text-right">
                                <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <Button variant="ghost" size="icon">
                                    <MoreHorizontal className="h-4 w-4" />
                                    </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                    <DropdownMenuItem onClick={() => handleEdit(coupon)}>
                                    <Edit className="mr-2 h-4 w-4" />
                                    Edit
                                    </DropdownMenuItem>
                                    <DropdownMenuItem className="text-destructive">
                                    <Trash2 className="mr-2 h-4 w-4" />
                                    Delete
                                    </DropdownMenuItem>
                                </DropdownMenuContent>
                                </DropdownMenu>
                            </TableCell>
                        </TableRow>
                    )
                  })}
                </TableBody>
              </Table>
            )}
            {!isLoading && !error && (!coupons || coupons.length === 0) && (
              <Alert>
                <Percent className="h-4 w-4" />
                <AlertTitle>No Coupons Found</AlertTitle>
                <AlertDescription>
                  There are no coupons to display. Click "Create New Coupon"
                  to get started.
                </AlertDescription>
              </Alert>
            )}
          </CardContent>
        </Card>

        <DialogContent className="sm:max-w-[600px]">
          <DialogHeader>
            <DialogTitle>
              {selectedCoupon ? 'Edit Coupon' : 'Create New Coupon'}
            </DialogTitle>
             <DialogDescription>
                {selectedCoupon ? `Editing the "${selectedCoupon.code}" coupon.` : 'Fill in the details to create a new coupon.'}
            </DialogDescription>
          </DialogHeader>
          <CouponForm
            coupon={selectedCoupon}
            onSave={() => setIsModalOpen(false)}
          />
        </DialogContent>
      </Dialog>
    </div>
  );
}
