
'use client';

import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import {
  Users,
  ShoppingCart,
  DollarSign,
  Clock,
  CheckCircle,
  XCircle,
  Archive,
} from 'lucide-react';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  ResponsiveContainer,
  Line,
  LineChart as RechartsLineChart,
} from 'recharts';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, collectionGroup, query, where } from 'firebase/firestore';
import type { Order, User, Product } from '@/lib/data';
import { useMemo } from 'react';
import { Skeleton } from '@/components/ui/skeleton';

export default function AdminDashboardPage() {
  const firestore = useFirestore();

  const usersCollectionRef = useMemoFirebase(
    () => collection(firestore, 'users'),
    [firestore]
  );
  const ordersCollectionRef = useMemoFirebase(
    () => collectionGroup(firestore, 'orders'),
    [firestore]
  );
  const productsCollectionRef = useMemoFirebase(
    () => collection(firestore, 'products'),
    [firestore]
  );

  const { data: users, isLoading: usersLoading } = useCollection<User>(usersCollectionRef);
  const { data: orders, isLoading: ordersLoading } = useCollection<Order>(ordersCollectionRef);
  const { data: products, isLoading: productsLoading } = useCollection<Product>(productsCollectionRef);

  const isLoading = usersLoading || ordersLoading || productsLoading;
  
  const dashboardData = useMemo(() => {
    if (!users || !orders || !products) {
      return {
        summaryCards: [],
        salesData: [],
        userGrowthData: [],
      };
    }
    
    // Process Sales Data
    const monthlySales = Array(6).fill(0).map(() => ({ sales: 0, orders: 0 }));
    const monthNames = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'];
    const currentYear = new Date().getFullYear();

    orders.forEach((order) => {
        const orderDate = new Date(order.orderDate);
        if (orderDate.getFullYear() === currentYear) {
            const month = orderDate.getMonth();
            if (month < 6) { // Only for first 6 months
                 // This is a simplification. Real revenue requires product prices.
                 // Assuming each order from mock data has one product.
                monthlySales[month].sales += order.product.price || 0;
                monthlySales[month].orders += 1;
            }
        }
    });

    const salesData = monthNames.map((name, index) => ({
      name,
      sales: monthlySales[index].sales,
    }));

    // Process User Growth
    const monthlyUsers = Array(6).fill(0);
     users.forEach((user) => {
        if (user.registeredDate?.seconds) {
            const regDate = new Date(user.registeredDate.seconds * 1000);
            if (regDate.getFullYear() === currentYear) {
                const month = regDate.getMonth();
                if (month < 6) {
                    monthlyUsers[month]++;
                }
            }
        }
    });

    const userGrowthData = monthNames.map((name, index) => ({
      name,
      users: monthlyUsers.reduce((acc, count, i) => i <= index ? acc + count : acc, 0),
    }));

    const totalRevenue = orders.reduce((acc, order) => acc + (order.product.price || 0), 0);
    const totalOrders = orders.length;
    const completedOrders = orders.filter(o => o.status === 'Delivered').length;
    const pendingOrders = orders.filter(o => o.status !== 'Delivered' && o.status !== 'Cancelled').length;
    const cancelledOrders = orders.filter(o => o.status === 'Cancelled').length;
    const totalUsers = users.length;
    const lowStockProducts = products.filter(p => p.stock !== undefined && p.stock < 10).length;


    const summaryCards = [
      { title: 'Total Revenue', value: `₹${totalRevenue.toFixed(2)}`, icon: DollarSign, change: '+20.1% from last month' },
      { title: 'Total Orders', value: totalOrders, icon: ShoppingCart, change: '+18.1% from last month' },
      { title: 'Completed Orders', value: completedOrders, icon: CheckCircle, change: '+19% from last month' },
      { title: 'Total Users', value: totalUsers, icon: Users, change: '+17.2% from last month' },
      { title: 'Pending Orders', value: pendingOrders, icon: Clock, change: '5 more than last week' },
      { title: 'Cancelled Orders', value: cancelledOrders, icon: XCircle, change: '2 less than last week' },
      { title: 'Today\'s Sales', value: '₹1,250.00', icon: DollarSign, change: '+8% from yesterday' }, // Static for now
      { title: 'Low Stock Products', value: lowStockProducts, icon: Archive, change: '2 more alerts' },
    ];
    
    return { summaryCards, salesData, userGrowthData };

  }, [users, orders, products]);
  
  if (isLoading) {
    return (
        <div className="p-4 md:p-8 space-y-6">
            <h1 className="text-3xl font-bold">Admin Dashboard</h1>
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                {[...Array(8)].map((_, i) => <Skeleton key={i} className="h-28 w-full" />)}
            </div>
             <div className="grid grid-cols-1 lg:grid-cols-5 gap-6">
                <Skeleton className="lg:col-span-3 h-[350px]" />
                <Skeleton className="lg:col-span-2 h-[350px]" />
            </div>
        </div>
    )
  }


  return (
    <div className="p-4 md:p-8 space-y-6">
      <h1 className="text-3xl font-bold">Admin Dashboard</h1>

      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {dashboardData.summaryCards.map((item, index) => (
          <Card key={index}>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">{item.title}</CardTitle>
              <item.icon className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{item.value}</div>
              <p className="text-xs text-muted-foreground">{item.change}</p>
            </CardContent>
          </Card>
        ))}
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-5 gap-6">
        <Card className="lg:col-span-3">
          <CardHeader>
            <CardTitle>Sales Overview</CardTitle>
          </CardHeader>
          <CardContent className="pl-2">
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={dashboardData.salesData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="name" stroke="#888888" fontSize={12} tickLine={false} axisLine={false} />
                <YAxis stroke="#888888" fontSize={12} tickLine={false} axisLine={false} tickFormatter={(value) => `₹${value}`} />
                <Tooltip
                  content={({ active, payload }) => {
                    if (active && payload && payload.length) {
                      return (
                        <div className="rounded-lg border bg-background p-2 shadow-sm">
                          <div className="grid grid-cols-2 gap-2">
                            <div className="flex flex-col">
                              <span className="text-[0.70rem] uppercase text-muted-foreground">
                                Month
                              </span>
                              <span className="font-bold text-muted-foreground">
                                {payload[0].payload.name}
                              </span>
                            </div>
                            <div className="flex flex-col">
                              <span className="text-[0.70rem] uppercase text-muted-foreground">
                                Sales
                              </span>
                              <span className="font-bold">
                                ₹{payload[0].value}
                              </span>
                            </div>
                          </div>
                        </div>
                      )
                    }
                    return null
                  }}
                />
                <Legend />
                <Bar dataKey="sales" fill="hsl(var(--primary))" radius={[4, 4, 0, 0]} />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>
        <Card className="lg:col-span-2">
           <CardHeader>
            <CardTitle>User Growth</CardTitle>
          </CardHeader>
          <CardContent className="pl-2">
             <ResponsiveContainer width="100%" height={300}>
                <RechartsLineChart data={dashboardData.userGrowthData}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="name" stroke="#888888" fontSize={12} tickLine={false} axisLine={false}/>
                    <YAxis stroke="#888888" fontSize={12} tickLine={false} axisLine={false} />
                    <Tooltip />
                    <Legend />
                    <Line type="monotone" dataKey="users" stroke="hsl(var(--primary))" />
                </RechartsLineChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
