
'use client';

import {
  SidebarProvider,
  Sidebar,
  SidebarHeader,
  SidebarContent,
  SidebarFooter,
  SidebarTrigger,
  SidebarMenu,
  SidebarMenuItem,
  SidebarMenuButton,
  SidebarInset,
  SidebarGroup,
  SidebarGroupLabel,
} from '@/components/ui/sidebar';
import {
  LayoutDashboard,
  Users,
  Package,
  LogOut,
  Home,
  Settings,
  ShoppingCart,
  LayoutList,
  Ticket,
  Percent,
  Truck,
  CreditCard,
  Shield,
  FileText,
  Bell,
  LineChart,
  Megaphone,
  Palette,
  PackageCheck,
  Undo2,
  Mail,
  Star,
  Search,
  Code,
  Server,
  Webhook,
} from 'lucide-react';
import { usePathname, useRouter } from 'next/navigation';
import { Logo } from '@/components/logo';
import { Button } from '@/components/ui/button';
import { useAuth, useUser } from '@/firebase';
import { signOut } from 'firebase/auth';
import { doc, getDoc, setDoc, getDocs, collection } from 'firebase/firestore';
import { useFirestore } from '@/firebase';
import { useEffect, useState } from 'react';
import { LoaderCircle } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';
import { errorEmitter } from '@/firebase/error-emitter';
import { FirestorePermissionError } from '@/firebase/errors';

const menuItems = [
  {
    group: 'Main',
    items: [
      { href: '/admin/dashboard', label: 'Dashboard', icon: LayoutDashboard },
      { href: '/admin/reports', label: 'Reports & Analytics', icon: LineChart },
    ],
  },
  {
    group: 'Store Management',
    items: [
      { href: '/admin/orders', label: 'Order Management', icon: ShoppingCart },
      { href: '/admin/products', label: 'Product Management', icon: Package },
      { href: '/admin/categories', label: 'Category Management', icon: LayoutList },
      { href: '/admin/inventory', label: 'Inventory Alerts', icon: PackageCheck },
      { href: '/admin/returns', label: 'Return Requests', icon: Undo2 },
    ],
  },
   {
    group: 'Users & Engagement',
    items: [
      { href: '/admin/users', label: 'User Management', icon: Users },
      { href: '/admin/reviews', label: 'Reviews & Ratings', icon: Star },
      { href: '/admin/newsletter', label: 'Newsletter Subscribers', icon: Mail },
    ],
  },
  {
    group: 'Marketing',
    items: [
       { href: '/admin/coupons', label: 'Coupon Management', icon: Percent },
       { href: '/admin/notifications', label: 'Notifications', icon: Bell },
       { href: '/admin/content', label: 'Content Management', icon: Palette },
    ],
  },
  {
    group: 'Configuration',
    items: [
      { href: '/admin/transactions', label: 'Payments & Transactions', icon: CreditCard },
      { href: '/admin/logistics', label: 'Delivery & Logistics', icon: Truck },
      { href: '/admin/settings', label: 'Website Settings', icon: Settings },
      { href: '/admin/policies', label: 'Policy Management', icon: FileText },
      { href: '/admin/seo', label: 'SEO Settings', icon: Search },
      { href: '/admin/security', label: 'Admin Security', icon: Shield },
    ],
  },
  {
    group: 'Developer',
    items: [
      { href: '/admin/dev/connect', label: 'Developer Connect', icon: Webhook },
    ],
  },
];

export default function AdminLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  const pathname = usePathname();
  const router = useRouter();
  const auth = useAuth();
  const firestore = useFirestore();
  const { user, isUserLoading } = useUser();
  const [isAdmin, setIsAdmin] = useState<boolean | null>(null);
  const { toast } = useToast();

  useEffect(() => {
    const checkAdminRole = async () => {
      if (isUserLoading || !firestore || !user) {
        return;
      }

      const adminRoleRef = doc(firestore, 'roles_admin', user.uid);
      const rolesCollectionRef = collection(firestore, 'roles_admin');

      // 1. Check if the user's admin document exists
      try {
        const adminDocSnap = await getDoc(adminRoleRef);
        if (adminDocSnap.exists()) {
          setIsAdmin(true); // User is a verified admin
          return;
        }
      } catch (error) {
        errorEmitter.emit(
          'permission-error',
          new FirestorePermissionError({
            path: adminRoleRef.path,
            operation: 'get',
          })
        );
        setIsAdmin(false); // Assume not admin on error
        return;
      }

      // 2. If not an admin, check if they should be the first one
      try {
        const rolesSnapshot = await getDocs(rolesCollectionRef);
        if (rolesSnapshot.empty) {
          // Collection is empty, try to make this user the first admin
          const newAdminData = { email: user.email };
          setDoc(adminRoleRef, newAdminData)
            .then(() => {
              setIsAdmin(true); // Grant access immediately
            })
            .catch(() => {
              errorEmitter.emit(
                'permission-error',
                new FirestorePermissionError({
                  path: adminRoleRef.path,
                  operation: 'create',
                  requestResourceData: newAdminData,
                })
              );
              setIsAdmin(false);
            });
        } else {
          // Admins exist, but this user is not one. Deny access.
          setIsAdmin(false);
        }
      } catch (error) {
        errorEmitter.emit(
          'permission-error',
          new FirestorePermissionError({
            path: rolesCollectionRef.path,
            operation: 'list',
          })
        );
        setIsAdmin(false);
      }
    };

    checkAdminRole();
  }, [user, isUserLoading, firestore]);

  useEffect(() => {
    if (isAdmin === false) {
      toast({
        variant: 'destructive',
        title: 'Access Denied',
        description: 'You do not have permission to access the admin panel.',
      });
      router.replace('/login');
    }
  }, [isAdmin, router, toast]);

  const handleLogout = async () => {
    if (auth) {
      await signOut(auth);
    }
    router.push('/login');
  };

  if (isUserLoading || isAdmin === null) {
    return (
      <div className="flex h-screen items-center justify-center bg-background">
        <div className="flex flex-col items-center gap-2">
          <LoaderCircle className="h-8 w-8 animate-spin text-primary" />
          <p className="text-muted-foreground">Verifying credentials...</p>
        </div>
      </div>
    );
  }

  return (
    <SidebarProvider>
      <Sidebar>
        <SidebarHeader>
          <div className="flex items-center gap-2">
            <Logo />
            <SidebarTrigger />
          </div>
        </SidebarHeader>
        <SidebarContent>
            {menuItems.map((group) => (
              <SidebarGroup key={group.group}>
                <SidebarGroupLabel>{group.group}</SidebarGroupLabel>
                 <SidebarMenu>
                  {group.items.map((item) => (
                    <SidebarMenuItem key={item.href}>
                      <SidebarMenuButton
                        asChild
                        isActive={pathname === item.href}
                      >
                        <a href={item.href}>
                          <item.icon />
                          <span>{item.label}</span>
                        </a>
                      </SidebarMenuButton>
                    </SidebarMenuItem>
                  ))}
                </SidebarMenu>
              </SidebarGroup>
            ))}
        </SidebarContent>
        <SidebarFooter>
           <SidebarMenu>
            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <a href="/">
                  <Home />
                  <span>Back to Home</span>
                </a>
              </SidebarMenuButton>
            </SidebarMenuItem>
            <SidebarMenuItem>
              <Button variant="ghost" className="w-full justify-start" onClick={handleLogout}>
                <LogOut className="mr-2 h-4 w-4" />
                Logout
              </Button>
            </SidebarMenuItem>
          </SidebarMenu>
        </SidebarFooter>
      </Sidebar>
      <SidebarInset>{children}</SidebarInset>
    </SidebarProvider>
  );
}
