'use client';

import { useState } from 'react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
  DialogFooter
} from '@/components/ui/dialog';
import {
  FileDown,
  Mail,
  Trash2,
  AlertTriangle,
  LoaderCircle,
  Send,
} from 'lucide-react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, deleteDoc, doc } from 'firebase/firestore';
import type { NewsletterEmail } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { useToast } from '@/hooks/use-toast';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';

export default function AdminNewsletterPage() {
  const [isDeleting, setIsDeleting] = useState<string | null>(null);
  const firestore = useFirestore();
  const { toast } = useToast();

  const subscribersCollectionRef = useMemoFirebase(
    () => collection(firestore, 'newsletter_emails'),
    [firestore]
  );
  const {
    data: subscribers,
    isLoading,
    error,
  } = useCollection<NewsletterEmail>(subscribersCollectionRef);

  const handleDelete = async (subscriberId: string) => {
    setIsDeleting(subscriberId);
    const docRef = doc(firestore, 'newsletter_emails', subscriberId);
    try {
      await deleteDoc(docRef);
      toast({
        title: 'Subscriber Removed',
        description: 'The email has been removed from the newsletter list.',
      });
    } catch (e: any) {
      toast({
        variant: 'destructive',
        title: 'Deletion Failed',
        description: 'Could not remove the subscriber. Please try again.',
      });
    } finally {
      setIsDeleting(null);
    }
  };
  
  const handleExport = () => {
    toast({
        title: 'Exporting Subscribers',
        description: "Your subscriber list is being prepared for download. (This is a demo feature)",
    })
  }
  
  const handleSendBroadcast = () => {
    toast({
        title: 'Broadcast Sent (Demo)',
        description: "Your email broadcast has been sent to all subscribers. (This is a demo feature)",
    })
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div className="flex flex-col md:flex-row justify-between md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold">Newsletter Subscribers</h1>
          <p className="text-muted-foreground">
            View, manage, and engage with your email subscribers.
          </p>
        </div>
         <div className="flex gap-2">
            <Dialog>
                <DialogTrigger asChild>
                    <Button>
                        <Send className="mr-2" />
                        Send Broadcast
                    </Button>
                </DialogTrigger>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Compose Email Broadcast</DialogTitle>
                        <DialogDescription>Send a promotional email to all subscribers. This is a demo and will not actually send emails.</DialogDescription>
                    </DialogHeader>
                    <div className="grid gap-4 py-4">
                        <div className="grid grid-cols-4 items-center gap-4">
                            <Label htmlFor="subject" className="text-right">Subject</Label>
                            <Input id="subject" placeholder="e.g., Summer Sale is Here!" className="col-span-3" />
                        </div>
                         <div className="grid grid-cols-4 items-start gap-4">
                            <Label htmlFor="body" className="text-right pt-2">Body</Label>
                            <Textarea id="body" placeholder="Your message..." className="col-span-3 min-h-32" />
                        </div>
                    </div>
                     <DialogFooter>
                        <Button onClick={handleSendBroadcast}>Send to All</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>
            <Button variant="outline" onClick={handleExport}>
              <FileDown className="mr-2" />
              Export List
            </Button>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>All Subscribers ({subscribers?.length || 0})</CardTitle>
          <CardDescription>
            A list of all users subscribed to the newsletter.
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading && (
            <div className="space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          )}
          {error && (
            <Alert variant="destructive">
              <AlertTriangle className="h-4 w-4" />
              <AlertTitle>Error</AlertTitle>
              <AlertDescription>
                Failed to load subscribers. Please ensure you have admin permissions.
              </AlertDescription>
            </Alert>
          )}
          {!isLoading && !error && subscribers && subscribers.length > 0 && (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Email Address</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {subscribers.map((subscriber) => (
                  <TableRow key={subscriber.id}>
                    <TableCell className="font-medium">{subscriber.email}</TableCell>
                    <TableCell className="text-right">
                       <AlertDialog>
                          <AlertDialogTrigger asChild>
                            <Button variant="ghost" size="icon" className="text-destructive hover:bg-destructive/10 hover:text-destructive" disabled={isDeleting === subscriber.id}>
                               {isDeleting === subscriber.id ? <LoaderCircle className="h-4 w-4 animate-spin" /> : <Trash2 className="h-4 w-4" />}
                            </Button>
                          </AlertDialogTrigger>
                          <AlertDialogContent>
                            <AlertDialogHeader>
                              <AlertDialogTitle>Are you sure?</AlertDialogTitle>
                              <AlertDialogDescription>
                                This will permanently remove the email "{subscriber.email}" from your subscriber list.
                              </AlertDialogDescription>
                            </AlertDialogHeader>
                            <AlertDialogFooter>
                              <AlertDialogCancel>Cancel</AlertDialogCancel>
                              <AlertDialogAction onClick={() => handleDelete(subscriber.id)} className="bg-destructive hover:bg-destructive/90">
                                Remove
                              </AlertDialogAction>
                            </AlertDialogFooter>
                          </AlertDialogContent>
                        </AlertDialog>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
          {!isLoading && !error && (!subscribers || subscribers.length === 0) && (
            <Alert>
              <Mail className="h-4 w-4" />
              <AlertTitle>No Subscribers Found</AlertTitle>
              <AlertDescription>
                There are no newsletter subscribers to display.
              </AlertDescription>
            </Alert>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
