
'use client';

import { useState } from 'react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
  CardFooter,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { Bell, Send, Megaphone, UserPlus, ShoppingCart, LoaderCircle, AlertTriangle } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';
import { Separator } from '@/components/ui/separator';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection } from 'firebase/firestore';
import type { AdminNotification } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { formatDistanceToNow } from 'date-fns';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';

const iconMap: { [key: string]: React.ElementType } = {
  UserPlus: UserPlus,
  ShoppingCart: ShoppingCart,
  default: Bell,
};


function AdminActivityFeed() {
    const firestore = useFirestore();

    const notificationsCollectionRef = useMemoFirebase(
        () => collection(firestore, 'admin_notifications'),
        [firestore]
    );

    const { data: notifications, isLoading, error } = useCollection<AdminNotification>(notificationsCollectionRef);

    return (
        <Card>
           <CardHeader>
            <CardTitle className="flex items-center gap-2">
                <Bell />
                Admin Activity Feed
            </CardTitle>
            <CardDescription>
              Recent important events in your store.
            </CardDescription>
          </CardHeader>
          <CardContent>
            {isLoading && (
                <div className="space-y-4">
                    <Skeleton className="h-12 w-full" />
                    <Skeleton className="h-12 w-full" />
                    <Skeleton className="h-12 w-full" />
                </div>
            )}
            {error && (
                <Alert variant="destructive">
                    <AlertTriangle className="h-4 w-4" />
                    <AlertTitle>Error</AlertTitle>
                    <AlertDescription>Failed to load notifications. Please try again.</AlertDescription>
                </Alert>
            )}
            {!isLoading && !error && notifications && (
                 <div className="space-y-4">
                    {notifications.length > 0 ? notifications.map((activity) => {
                        const IconComponent = iconMap[activity.icon] || iconMap.default;
                        return (
                            <div key={activity.id} className="flex items-start gap-3">
                                <IconComponent className="h-5 w-5 text-muted-foreground mt-1 flex-shrink-0" />
                                <div className="flex-1">
                                    <p className="text-sm">{activity.text}</p>
                                    <p className="text-xs text-muted-foreground">
                                        {activity.timestamp ? formatDistanceToNow(activity.timestamp.toDate(), { addSuffix: true }) : 'just now'}
                                    </p>
                                </div>
                            </div>
                        )
                    }) : (
                        <Alert>
                            <Bell className="h-4 w-4" />
                            <AlertTitle>No Activities</AlertTitle>
                            <AlertDescription>There are no admin notifications to display yet.</AlertDescription>
                        </Alert>
                    )}
                </div>
            )}
          </CardContent>
        </Card>
    )
}


export default function AdminNotificationsPage() {
  const { toast } = useToast();
  const [isSending, setIsSending] = useState(false);

  const handleSendNotification = () => {
    setIsSending(true);
    setTimeout(() => {
        toast({
            title: "Push Notification Sent (Demo)",
            description: "Your notification has been dispatched to the selected users."
        });
        setIsSending(false);
    }, 1500)
  }

  const handleUpdateBanner = () => {
    toast({
        title: "Announcement Banner Updated",
        description: "The site-wide announcement banner has been updated."
    })
  }


  return (
    <div className="p-4 md:p-8 space-y-6">
      <div className="flex flex-col md:flex-row justify-between md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold">Notifications</h1>
          <p className="text-muted-foreground">
            Engage with users and manage site-wide announcements.
          </p>
        </div>
      </div>
      
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">

        {/* Column 1: Push Notifications */}
        <Card className="lg:col-span-2">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
                <Send />
                Push Notification Composer
            </CardTitle>
            <CardDescription>
              Send a targeted message directly to your users' devices. (Demo feature)
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
                <Label htmlFor="title">Title</Label>
                <Input id="title" placeholder="e.g., Weekend Flash Sale!" />
            </div>
             <div className="space-y-2">
                <Label htmlFor="message">Message</Label>
                <Textarea id="message" placeholder="Your notification body..." />
            </div>
             <div className="space-y-2">
                <Label htmlFor="audience">Target Audience</Label>
                <Select>
                    <SelectTrigger>
                        <SelectValue placeholder="Select audience..." />
                    </SelectTrigger>
                    <SelectContent>
                        <SelectItem value="all">All Users</SelectItem>
                        <SelectItem value="new">New Users (Last 7 Days)</SelectItem>
                        <SelectItem value="inactive">Inactive Users (30+ Days)</SelectItem>
                    </SelectContent>
                </Select>
            </div>
          </CardContent>
          <CardFooter>
            <Button onClick={handleSendNotification} disabled={isSending}>
                {isSending && <LoaderCircle className="mr-2 animate-spin" />}
                Send Notification
            </Button>
          </CardFooter>
        </Card>

        {/* Column 2: Admin Activity Feed */}
        <AdminActivityFeed />
      </div>

       {/* Announcement Banner Management */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Megaphone />
            Announcement Banner
          </CardTitle>
          <CardDescription>
            Display a site-wide banner for promotions or important notices.
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
            <div className="space-y-2">
                <Label htmlFor="banner-text">Banner Message</Label>
                <Input id="banner-text" placeholder="e.g., Free shipping on all orders over ₹500!" />
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                 <div className="space-y-2">
                    <Label htmlFor="banner-style">Banner Style</Label>
                    <Select defaultValue="info">
                        <SelectTrigger>
                            <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                            <SelectItem value="info">Info (Blue)</SelectItem>
                            <SelectItem value="warning">Warning (Amber)</SelectItem>
                            <SelectItem value="success">Success (Green)</SelectItem>
                        </SelectContent>
                    </Select>
                </div>
                 <div className="flex items-center space-x-2 pt-6">
                    <Switch id="banner-enabled" />
                    <Label htmlFor="banner-enabled">Enable Banner</Label>
                </div>
            </div>
        </CardContent>
        <CardFooter>
            <Button onClick={handleUpdateBanner}>Update Banner</Button>
        </CardFooter>
      </Card>
    </div>
  );
}

    