
'use client';

import React from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import {
  Card,
  CardHeader,
  CardTitle,
  CardContent,
  CardDescription,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Textarea } from '@/components/ui/textarea';
import { useToast } from '@/hooks/use-toast';
import { useFirestore, useDoc, useMemoFirebase } from '@/firebase';
import { doc } from 'firebase/firestore';
import { LoaderCircle, FileText } from 'lucide-react';
import type { Settings as SettingsData } from '@/lib/data';
import { updateSettings } from '@/firebase/firestore/settings';
import { Skeleton } from '@/components/ui/skeleton';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';

const policySchema = z.object({
  privacyPolicy: z.string().optional(),
  termsAndConditions: z.string().optional(),
  returnPolicy: z.string().optional(),
});

type PolicyFormValues = z.infer<typeof policySchema>;

export default function AdminPoliciesPage() {
  const { toast } = useToast();
  const firestore = useFirestore();
  const settingsDocRef = useMemoFirebase(
    () => doc(firestore, 'settings', 'main'),
    [firestore]
  );

  const { data: settings, isLoading } = useDoc<SettingsData>(settingsDocRef);

  const form = useForm<PolicyFormValues>({
    resolver: zodResolver(policySchema),
    defaultValues: {
      privacyPolicy: '',
      termsAndConditions: '',
      returnPolicy: '',
    },
  });

  React.useEffect(() => {
    if (settings) {
      form.reset({
        privacyPolicy: settings.privacyPolicy,
        termsAndConditions: settings.termsAndConditions,
        returnPolicy: settings.returnPolicy,
      });
    }
  }, [settings, form]);

  const onSubmit = async (values: PolicyFormValues) => {
    try {
      await updateSettings(firestore, values);
      toast({
        title: 'Policies Saved',
        description: 'Your legal policies have been successfully updated.',
      });
    } catch (e) {
      toast({
        variant: 'destructive',
        title: 'Save Failed',
        description: 'Could not save policies. Please try again.',
      });
    }
  };
  
  const isSubmitting = form.formState.isSubmitting;

  if (isLoading) {
    return (
      <div className="p-4 md:p-8 space-y-6">
        <Skeleton className="h-8 w-64" />
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-48" />
            <Skeleton className="h-4 w-full max-w-sm mt-2" />
          </CardHeader>
          <CardContent className="space-y-6">
            <Skeleton className="h-40 w-full" />
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div>
        <h1 className="text-3xl font-bold">Policy Management</h1>
        <p className="text-muted-foreground">
          Update your website's legal documents and user policies.
        </p>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
          <Tabs defaultValue="privacy">
             <TabsList>
              <TabsTrigger value="privacy">Privacy Policy</TabsTrigger>
              <TabsTrigger value="terms">Terms & Conditions</TabsTrigger>
              <TabsTrigger value="returns">Return Policy</TabsTrigger>
            </TabsList>
            <TabsContent value="privacy" className="mt-4">
               <Card>
                 <CardHeader>
                    <CardTitle>Privacy Policy</CardTitle>
                    <CardDescription>Manage the content of your privacy policy page.</CardDescription>
                 </CardHeader>
                 <CardContent>
                     <FormField
                        control={form.control}
                        name="privacyPolicy"
                        render={({ field }) => (
                        <FormItem>
                            <FormControl>
                            <Textarea
                                placeholder="Enter your privacy policy text here..."
                                className="min-h-96 font-mono"
                                {...field}
                            />
                            </FormControl>
                            <FormMessage />
                        </FormItem>
                        )}
                    />
                 </CardContent>
               </Card>
            </TabsContent>
             <TabsContent value="terms" className="mt-4">
               <Card>
                 <CardHeader>
                    <CardTitle>Terms & Conditions</CardTitle>
                    <CardDescription>Manage the content of your terms and conditions page.</CardDescription>
                 </CardHeader>
                 <CardContent>
                     <FormField
                        control={form.control}
                        name="termsAndConditions"
                        render={({ field }) => (
                        <FormItem>
                            <FormControl>
                            <Textarea
                                placeholder="Enter your terms and conditions text here..."
                                className="min-h-96 font-mono"
                                {...field}
                            />
                            </FormControl>
                            <FormMessage />
                        </FormItem>
                        )}
                    />
                 </CardContent>
               </Card>
            </TabsContent>
            <TabsContent value="returns" className="mt-4">
               <Card>
                 <CardHeader>
                    <CardTitle>Return & Refund Policy</CardTitle>
                    <CardDescription>Manage the content of your return and refund policy.</CardDescription>
                 </CardHeader>
                 <CardContent>
                     <FormField
                        control={form.control}
                        name="returnPolicy"
                        render={({ field }) => (
                        <FormItem>
                            <FormControl>
                            <Textarea
                                placeholder="Enter your return policy text here..."
                                className="min-h-96 font-mono"
                                {...field}
                            />
                            </FormControl>
                            <FormMessage />
                        </FormItem>
                        )}
                    />
                 </CardContent>
               </Card>
            </TabsContent>
          </Tabs>

          <div className="flex justify-end pt-4">
            <Button type="submit" disabled={isSubmitting}>
              {isSubmitting && <LoaderCircle className="animate-spin mr-2" />}
              Save All Policies
            </Button>
          </div>
        </form>
      </Form>
    </div>
  );
}
