'use client';

import { useState, useMemo } from 'react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import {
  CheckCircle,
  XCircle,
  Trash2,
  AlertTriangle,
  Star,
  LoaderCircle,
  MoreHorizontal,
  User,
} from 'lucide-react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, doc, updateDoc, deleteDoc } from 'firebase/firestore';
import type { Review, ReviewStatus } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { useToast } from '@/hooks/use-toast';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import Link from 'next/link';

type ReviewWithUserDetails = Review & { userName?: string };

const statusVariant: {
  [key in ReviewStatus]: 'default' | 'secondary' | 'destructive' | 'outline';
} = {
  pending: 'secondary',
  approved: 'default',
  rejected: 'destructive',
};

function ReviewsTable({
  reviews,
  onUpdateStatus,
  onDelete,
  isUpdating,
}: {
  reviews: ReviewWithUserDetails[];
  onUpdateStatus: (reviewId: string, status: ReviewStatus) => void;
  onDelete: (reviewId: string) => void;
  isUpdating: (reviewId: string) => boolean;
}) {
  return (
    <Table>
      <TableHeader>
        <TableRow>
          <TableHead>Product</TableHead>
          <TableHead>User</TableHead>
          <TableHead>Rating</TableHead>
          <TableHead>Comment</TableHead>
          <TableHead>Status</TableHead>
          <TableHead className="text-right">Actions</TableHead>
        </TableRow>
      </TableHeader>
      <TableBody>
        {reviews.map((review) => (
          <TableRow key={review.id}>
            <TableCell className="font-medium">{review.productId}</TableCell>
            <TableCell>{review.userName || review.userId}</TableCell>
            <TableCell>
              <div className="flex items-center">
                {review.rating} <Star className="ml-1 h-4 w-4 text-yellow-400 fill-yellow-400" />
              </div>
            </TableCell>
            <TableCell className="max-w-xs truncate">{review.comment}</TableCell>
            <TableCell>
              <Badge variant={statusVariant[review.status]}>{review.status}</Badge>
            </TableCell>
            <TableCell className="text-right">
              {isUpdating(review.id) ? (
                <LoaderCircle className="h-4 w-4 animate-spin ml-auto" />
              ) : (
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="ghost" size="icon">
                      <MoreHorizontal className="h-4 w-4" />
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end">
                    {review.status !== 'approved' && (
                      <DropdownMenuItem onClick={() => onUpdateStatus(review.id, 'approved')}>
                        <CheckCircle className="mr-2 h-4 w-4 text-green-500" />
                        Approve
                      </DropdownMenuItem>
                    )}
                    {review.status !== 'rejected' && (
                      <DropdownMenuItem onClick={() => onUpdateStatus(review.id, 'rejected')}>
                        <XCircle className="mr-2 h-4 w-4 text-red-500" />
                        Reject
                      </DropdownMenuItem>
                    )}
                    <DropdownMenuItem asChild>
                      <Link href={`/admin/users/${review.userId}`}>
                        <User className="mr-2 h-4 w-4" />
                        View/Block User
                      </Link>
                    </DropdownMenuItem>
                    <DropdownMenuItem
                      onClick={() => onDelete(review.id)}
                      className="text-destructive"
                    >
                      <Trash2 className="mr-2 h-4 w-4" />
                      Delete
                    </DropdownMenuItem>
                  </DropdownMenuContent>
                </DropdownMenu>
              )}
            </TableCell>
          </TableRow>
        ))}
      </TableBody>
    </Table>
  );
}

export default function AdminReviewsPage() {
  const firestore = useFirestore();
  const { toast } = useToast();
  const [updatingIds, setUpdatingIds] = useState<string[]>([]);

  const reviewsCollectionRef = useMemoFirebase(
    () => collection(firestore, 'reviews'),
    [firestore]
  );
  const {
    data: reviews,
    isLoading,
    error,
  } = useCollection<ReviewWithUserDetails>(reviewsCollectionRef);

  const filteredReviews = useMemo(() => {
    const filterByStatus = (status: ReviewStatus) => reviews?.filter((r) => r.status === status) || [];
    return {
      pending: filterByStatus('pending'),
      approved: filterByStatus('approved'),
      rejected: filterByStatus('rejected'),
    };
  }, [reviews]);
  
  const handleUpdateStatus = async (reviewId: string, status: ReviewStatus) => {
    setUpdatingIds((prev) => [...prev, reviewId]);
    const reviewDocRef = doc(firestore, 'reviews', reviewId);
    try {
      await updateDoc(reviewDocRef, { status });
      toast({
        title: 'Review Updated',
        description: `The review has been ${status}.`,
      });
    } catch (error) {
      toast({
        variant: 'destructive',
        title: 'Update Failed',
        description: 'Could not update the review status.',
      });
    } finally {
      setUpdatingIds((prev) => prev.filter((id) => id !== reviewId));
    }
  };

  const handleDelete = async (reviewId: string) => {
    setUpdatingIds((prev) => [...prev, reviewId]);
    const reviewDocRef = doc(firestore, 'reviews', reviewId);
    try {
      await deleteDoc(reviewDocRef);
      toast({
        title: 'Review Deleted',
        description: 'The review has been permanently removed.',
      });
    } catch (error) {
      toast({
        variant: 'destructive',
        title: 'Deletion Failed',
        description: 'Could not delete the review.',
      });
    } finally {
      setUpdatingIds((prev) => prev.filter((id) => id !== reviewId));
    }
  };
  
  const isUpdating = (id: string) => updatingIds.includes(id);

  if (isLoading) {
    return (
      <div className="p-4 md:p-8 space-y-6">
        <Skeleton className="h-10 w-64" />
        <Skeleton className="h-12 w-full" />
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-48" />
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <Skeleton className="h-12 w-full" />
              <Skeleton className="h-12 w-full" />
              <Skeleton className="h-12 w-full" />
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-4 md:p-8">
        <Alert variant="destructive">
          <AlertTriangle className="h-4 w-4" />
          <AlertTitle>Error</AlertTitle>
          <AlertDescription>Failed to load reviews. Please try again.</AlertDescription>
        </Alert>
      </div>
    );
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div>
        <h1 className="text-3xl font-bold">Reviews & Ratings</h1>
        <p className="text-muted-foreground">Approve, reject, and manage product reviews.</p>
      </div>

      <Tabs defaultValue="pending">
        <TabsList>
          <TabsTrigger value="pending">Pending ({filteredReviews.pending.length})</TabsTrigger>
          <TabsTrigger value="approved">Approved ({filteredReviews.approved.length})</TabsTrigger>
          <TabsTrigger value="rejected">Rejected ({filteredReviews.rejected.length})</TabsTrigger>
        </TabsList>
        <TabsContent value="pending">
          <Card>
            <CardHeader>
              <CardTitle>Pending Reviews</CardTitle>
              <CardDescription>Reviews awaiting moderation.</CardDescription>
            </CardHeader>
            <CardContent>
              {filteredReviews.pending.length > 0 ? (
                <ReviewsTable reviews={filteredReviews.pending} onUpdateStatus={handleUpdateStatus} onDelete={handleDelete} isUpdating={isUpdating} />
              ) : (
                <Alert>
                  <AlertTitle>No Pending Reviews</AlertTitle>
                  <AlertDescription>There are no reviews awaiting moderation.</AlertDescription>
                </Alert>
              )}
            </CardContent>
          </Card>
        </TabsContent>
        <TabsContent value="approved">
          <Card>
             <CardHeader>
              <CardTitle>Approved Reviews</CardTitle>
              <CardDescription>Reviews currently visible to customers.</CardDescription>
            </CardHeader>
            <CardContent>
              {filteredReviews.approved.length > 0 ? (
                <ReviewsTable reviews={filteredReviews.approved} onUpdateStatus={handleUpdateStatus} onDelete={handleDelete} isUpdating={isUpdating} />
              ) : (
                <Alert>
                  <AlertTitle>No Approved Reviews</AlertTitle>
                  <AlertDescription>No reviews have been approved yet.</AlertDescription>
                </Alert>
              )}
            </CardContent>
          </Card>
        </TabsContent>
        <TabsContent value="rejected">
          <Card>
            <CardHeader>
              <CardTitle>Rejected Reviews</CardTitle>
              <CardDescription>Reviews that have been rejected and are not visible.</CardDescription>
            </CardHeader>
            <CardContent>
              {filteredReviews.rejected.length > 0 ? (
                <ReviewsTable reviews={filteredReviews.rejected} onUpdateStatus={handleUpdateStatus} onDelete={handleDelete} isUpdating={isUpdating} />
              ) : (
                <Alert>
                  <AlertTitle>No Rejected Reviews</AlertTitle>
                  <AlertDescription>No reviews have been rejected yet.</AlertDescription>
                </Alert>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
