
'use client';

import { useState } from 'react';
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
  CardFooter,
} from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
  DialogFooter
} from '@/components/ui/dialog';
import {
  Shield,
  PlusCircle,
  KeyRound,
  Trash2,
  List,
  Fingerprint,
  AlertTriangle,
  LoaderCircle,
  MoreHorizontal
} from 'lucide-react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, doc, setDoc, deleteDoc, query, orderBy, limit } from 'firebase/firestore';
import type { Admin, AdminLoginLog } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { useToast } from '@/hooks/use-toast';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { Badge } from '@/components/ui/badge';
import { errorEmitter } from '@/firebase/error-emitter';
import { FirestorePermissionError } from '@/firebase/errors';
import { formatDistanceToNow } from 'date-fns';

function AdminLoginLogs() {
  const firestore = useFirestore();
  const logsQuery = useMemoFirebase(
    () => query(collection(firestore, 'admin_login_logs'), orderBy('timestamp', 'desc'), limit(10)),
    [firestore]
  );
  
  const { data: logs, isLoading, error } = useCollection<AdminLoginLog>(logsQuery);

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2"><List /> Admin Login Logs</CardTitle>
        <CardDescription>A log of recent administrator login attempts.</CardDescription>
      </CardHeader>
      <CardContent>
         {isLoading && (
            <div className="space-y-2">
                <Skeleton className="h-10 w-full" />
                <Skeleton className="h-10 w-full" />
                <Skeleton className="h-10 w-full" />
            </div>
         )}
         {error && (
             <Alert variant="destructive">
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle>Error Loading Logs</AlertTitle>
                <AlertDescription>Could not fetch login logs. Please try again.</AlertDescription>
              </Alert>
         )}
         {!isLoading && !error && logs && logs.length > 0 && (
             <Table>
                <TableHeader>
                    <TableRow>
                        <TableHead>Email</TableHead>
                        <TableHead>Status</TableHead>
                        <TableHead>IP Address</TableHead>
                        <TableHead>Time</TableHead>
                    </TableRow>
                </TableHeader>
                <TableBody>
                    {logs.map(log => (
                        <TableRow key={log.id}>
                            <TableCell>{log.email}</TableCell>
                            <TableCell>
                                <Badge variant={log.status === 'success' ? 'default' : 'destructive'}>
                                    {log.status}
                                </Badge>
                            </TableCell>
                            <TableCell className="font-mono">{log.ipAddress}</TableCell>
                            <TableCell>{formatDistanceToNow(log.timestamp.toDate(), { addSuffix: true })}</TableCell>
                        </TableRow>
                    ))}
                </TableBody>
             </Table>
         )}
         {!isLoading && !error && (!logs || logs.length === 0) && (
            <Alert>
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle>No Logs Found</AlertTitle>
                <AlertDescription>There are no admin login logs to display yet.</AlertDescription>
            </Alert>
         )}
      </CardContent>
    </Card>
  )
}


export default function AdminSecurityPage() {
  const [isAddModalOpen, setIsAddModalOpen] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [newAdminEmail, setNewAdminEmail] = useState('');
  const firestore = useFirestore();
  const { toast } = useToast();

  const adminRolesCollectionRef = useMemoFirebase(
    () => collection(firestore, 'roles_admin'),
    [firestore]
  );

  const { data: admins, isLoading, error } = useCollection<Admin>(adminRolesCollectionRef);
  
  const handleAddAdmin = async () => {
    if (!newAdminEmail) {
        toast({ variant: 'destructive', title: 'Email is required.' });
        return;
    }
    setIsSubmitting(true);
    // In a real app, you'd want to find the user's UID from their email via a backend function.
    // For this client-side example, we'll use the email as a temporary ID, which is not secure.
    // This will likely fail security rules if they require the doc ID to be a UID.
    // The admin layout's logic is what creates the first admin securely. This is for adding subsequent admins.
    const tempDocId = newAdminEmail.replace(/[^a-zA-Z0-9]/g, '_');
    const newAdminRef = doc(firestore, 'roles_admin', tempDocId);
    
    setDoc(newAdminRef, { email: newAdminEmail })
    .then(() => {
        toast({
            title: 'Admin Added (Demo)',
            description: `${newAdminEmail} has been granted admin privileges. They must already have a user account.`,
        });
        setIsAddModalOpen(false);
        setNewAdminEmail('');
    })
    .catch((e) => {
         errorEmitter.emit(
            'permission-error',
            new FirestorePermissionError({
                path: newAdminRef.path,
                operation: 'create',
                requestResourceData: { email: newAdminEmail },
            })
        );
    })
    .finally(() => {
        setIsSubmitting(false);
    });
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div className="flex flex-col md:flex-row justify-between md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold">Admin Security</h1>
          <p className="text-muted-foreground">Manage admin accounts, roles, and security settings.</p>
        </div>
      </div>
      
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2"><Shield /> Admin Accounts</CardTitle>
          <CardDescription>View, add, or remove administrators.</CardDescription>
        </CardHeader>
        <CardContent>
           {isLoading && (
             <div className="space-y-2">
                <Skeleton className="h-12 w-full" />
                <Skeleton className="h-12 w-full" />
            </div>
          )}
          {error && (
             <Alert variant="destructive">
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle>Error Loading Admins</AlertTitle>
                <AlertDescription>
                  Could not fetch admin data. Please ensure you have the correct permissions.
                </AlertDescription>
              </Alert>
          )}
          {!isLoading && !error && admins && (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Email</TableHead>
                  <TableHead>Role</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {admins.map(admin => (
                  <TableRow key={admin.id}>
                    <TableCell className="font-medium">{admin.email}</TableCell>
                    <TableCell><Badge>Super Admin</Badge></TableCell>
                    <TableCell className="text-right">
                       <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" size="icon">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                             <DropdownMenuItem disabled>
                              <KeyRound className="mr-2 h-4 w-4" />
                              Change Role
                            </DropdownMenuItem>
                             <DropdownMenuItem className="text-destructive" disabled>
                              <Trash2 className="mr-2 h-4 w-4" />
                              Remove
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
        <CardFooter>
            <Dialog open={isAddModalOpen} onOpenChange={setIsAddModalOpen}>
                <DialogTrigger asChild>
                    <Button>
                        <PlusCircle className="mr-2" />
                        Add New Admin
                    </Button>
                </DialogTrigger>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Add New Administrator</DialogTitle>
                        <DialogDescription>
                            Enter the email address of the user you want to grant admin privileges to. They must have an existing user account.
                        </DialogDescription>
                    </DialogHeader>
                    <div className="space-y-2">
                        <Label htmlFor="email">User Email</Label>
                        <Input 
                            id="email" 
                            type="email" 
                            placeholder="user@example.com" 
                            value={newAdminEmail}
                            onChange={(e) => setNewAdminEmail(e.target.value)}
                        />
                    </div>
                    <DialogFooter>
                        <Button variant="ghost" onClick={() => setIsAddModalOpen(false)}>Cancel</Button>
                        <Button onClick={handleAddAdmin} disabled={isSubmitting}>
                            {isSubmitting && <LoaderCircle className="mr-2 h-4 w-4 animate-spin" />}
                            Grant Access
                        </Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>
        </CardFooter>
      </Card>
      
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <AdminLoginLogs />
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2"><Fingerprint /> Multi-Factor Authentication (2FA)</CardTitle>
            <CardDescription>Enhance security by requiring a second factor of authentication at login.</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between p-4 rounded-lg bg-muted/50">
              <div className="space-y-1">
                <h3 className="font-medium">Enable 2FA for all Admins</h3>
                <p className="text-sm text-muted-foreground">Admins will be required to use an authenticator app.</p>
              </div>
              <Switch disabled />
            </div>
             <Alert className="mt-4">
              <AlertTriangle className="h-4 w-4" />
              <AlertTitle>Feature Not Implemented</AlertTitle>
              <AlertDescription>2FA functionality is planned for a future update.</AlertDescription>
            </Alert>
          </CardContent>
        </Card>
      </div>

    </div>
  );
}

    