'use client';

import React, { useEffect } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import {
  Card,
  CardHeader,
  CardTitle,
  CardContent,
  CardDescription,
  CardFooter,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { useToast } from '@/hooks/use-toast';
import { useFirestore, useDoc, useMemoFirebase } from '@/firebase';
import { doc } from 'firebase/firestore';
import { LoaderCircle, Search, FileCode, ImageIcon } from 'lucide-react';
import type { Settings as SettingsData } from '@/lib/data';
import { updateSettings } from '@/firebase/firestore/settings';
import { Skeleton } from '@/components/ui/skeleton';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
  FormDescription,
} from '@/components/ui/form';

const seoSettingsSchema = z.object({
  metaTitle: z.string().max(60, 'Title should be 60 characters or less.').optional(),
  metaDescription: z.string().max(160, 'Description should be 160 characters or less.').optional(),
  ogImageUrl: z.string().url('Please enter a valid URL.').optional().or(z.literal('')),
  websiteKeywords: z.string().optional(),
});

type SeoSettingsFormValues = z.infer<typeof seoSettingsSchema>;

export default function AdminSeoPage() {
  const { toast } = useToast();
  const firestore = useFirestore();
  const settingsDocRef = useMemoFirebase(
    () => doc(firestore, 'settings', 'main'),
    [firestore]
  );

  const { data: settings, isLoading } = useDoc<SettingsData>(settingsDocRef);

  const form = useForm<SeoSettingsFormValues>({
    resolver: zodResolver(seoSettingsSchema),
    defaultValues: {
      metaTitle: '',
      metaDescription: '',
      ogImageUrl: '',
      websiteKeywords: '',
    },
  });

  useEffect(() => {
    if (settings) {
      form.reset({
        metaTitle: settings.metaTitle,
        metaDescription: settings.metaDescription,
        ogImageUrl: settings.ogImageUrl,
        websiteKeywords: settings.websiteKeywords,
      });
    }
  }, [settings, form]);

  const onSubmit = async (values: SeoSettingsFormValues) => {
    try {
      await updateSettings(firestore, values);
      toast({
        title: 'SEO Settings Saved',
        description: 'Your SEO settings have been successfully updated.',
      });
    } catch (e) {
      toast({
        variant: 'destructive',
        title: 'Save Failed',
        description: 'Could not save SEO settings. Please try again.',
      });
    }
  };
  
  const handleSitemapGeneration = () => {
    toast({
        title: 'Sitemap Generation Initiated',
        description: 'The sitemap is being generated in the background. This can take a few minutes.'
    });
  };

  const isSubmitting = form.formState.isSubmitting;

  if (isLoading) {
    return (
      <div className="p-4 md:p-8 space-y-6">
        <Skeleton className="h-8 w-64" />
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-48" />
            <Skeleton className="h-4 w-full max-w-sm mt-2" />
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="space-y-2">
              <Skeleton className="h-4 w-24" />
              <Skeleton className="h-10 w-full" />
            </div>
            <div className="space-y-2">
              <Skeleton className="h-4 w-24" />
              <Skeleton className="h-20 w-full" />
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div>
        <h1 className="text-3xl font-bold">SEO Settings</h1>
        <p className="text-muted-foreground">
          Manage your website's search engine optimization settings.
        </p>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Search />
                Meta Tags
              </CardTitle>
              <CardDescription>
                These tags help search engines understand your website's content.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <FormField
                control={form.control}
                name="metaTitle"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Global Meta Title</FormLabel>
                    <FormControl>
                      <Input
                        placeholder="Your Website Name - Your Tagline"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      The title that appears in browser tabs and search results. (Max 60 chars)
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="metaDescription"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Global Meta Description</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Describe what your website is about."
                        {...field}
                      />
                    </FormControl>
                     <FormDescription>
                      A brief summary of your page's content. (Max 160 chars)
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
               <FormField
                control={form.control}
                name="websiteKeywords"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Website Keywords</FormLabel>
                    <FormControl>
                      <Input
                        placeholder="pharmacy, medicine, health, wellness"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Comma-separated keywords relevant to your website.
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

           <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <ImageIcon />
                Social Sharing (Open Graph)
              </CardTitle>
              <CardDescription>
                Control how your website appears when shared on social media.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <FormField
                    control={form.control}
                    name="ogImageUrl"
                    render={({ field }) => (
                    <FormItem>
                        <FormLabel>OG Image URL</FormLabel>
                        <FormControl>
                        <Input
                            type="url"
                            placeholder="https://example.com/social-image.jpg"
                            {...field}
                        />
                        </FormControl>
                        <FormDescription>
                        A direct link to an image for social media previews (e.g., 1200x630px).
                        </FormDescription>
                        <FormMessage />
                    </FormItem>
                    )}
                />
            </CardContent>
          </Card>
          
           <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileCode />
                Sitemap
              </CardTitle>
              <CardDescription>
                Manage your website's sitemap for search engine indexing.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <p className="text-sm text-muted-foreground">
                    A sitemap helps search engines find and crawl all of your site's pages. You can trigger a manual regeneration here.
                </p>
            </CardContent>
             <CardFooter>
              <Button type="button" variant="outline" onClick={handleSitemapGeneration}>
                Generate Sitemap
              </Button>
            </CardFooter>
          </Card>

          <div className="flex justify-end">
            <Button type="submit" disabled={isSubmitting}>
              {isSubmitting && (
                <LoaderCircle className="animate-spin mr-2" />
              )}
              Save SEO Settings
            </Button>
          </div>
        </form>
      </Form>
    </div>
  );
}
