
'use client';

import React, { useEffect, useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import {
  Card,
  CardHeader,
  CardTitle,
  CardContent,
  CardDescription,
  CardFooter,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useToast } from '@/hooks/use-toast';
import { useFirestore, useDoc, useMemoFirebase } from '@/firebase';
import { doc } from 'firebase/firestore';
import { LoaderCircle, Palette, Brush, CreditCard, Shield, BarChart, Server, Download, HardHat, LayoutTemplate, Code, Image as ImageIcon } from 'lucide-react';
import type { Settings as SettingsData } from '@/lib/data';
import { updateSettings } from '@/firebase/firestore/settings';
import { Skeleton } from '@/components/ui/skeleton';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from '@/components/ui/form';
import { Switch } from '@/components/ui/switch';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Separator } from '@/components/ui/separator';
import { Textarea } from '@/components/ui/textarea';
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';
import { HomepageSectionsManager } from '@/components/homepage-sections-manager';
import Image from 'next/image';

const hexColorRegex = /^#[0-9a-fA-F]{6}$/;

const settingsSchema = z.object({
  websiteName: z.string().min(2, 'Website name is required.'),
  websiteTagline: z.string().optional(),
  logoUrl: z.string().url('Please enter a valid URL.').optional().or(z.literal('')),
  logoIcon: z.string().optional(),
  
  // Theme Colors
  background: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  foreground: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  card: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  primary: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  primaryForeground: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  secondary: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  secondaryForeground: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  destructive: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  destructiveForeground: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  ring: z.string().regex(hexColorRegex, 'Must be a valid hex color.').optional(),
  
  // Payment
  stripeApiKey: z.string().optional(),
  paypalClientId: z.string().optional(),
  razorpayApiKey: z.string().optional(),
  enableStripe: z.boolean().default(false),
  enablePaypal: z.boolean().default(false),
  enableRazorpay: z.boolean().default(false),
  defaultCurrency: z.string().min(3, "Please select a currency."),
  paymentTestMode: z.boolean().default(true),

  // API & Security
  publicApiKey: z.string().optional(),
  ipWhitelist: z.string().optional(),
  
  // Integrations
  googleAnalyticsId: z.string().optional(),
  facebookPixelId: z.string().optional(),
  googleTagManagerId: z.string().optional(),
  
  // Performance
  enableCache: z.boolean().default(true),
  imageCompression: z.enum(['low', 'medium', 'high']).default('medium'),
  
  // Maintenance
  maintenanceMode: z.boolean().default(false),
  maintenancePageBanner: z.string().optional(),

  homepageSections: z.array(z.object({
    id: z.string(),
    name: z.string(),
    enabled: z.boolean(),
  })).optional(),

  // Custom Code
  headerScripts: z.string().optional(),
  bodyStartScripts: z.string().optional(),
  footerScripts: z.string().optional(),
});

type SettingsFormValues = z.infer<typeof settingsSchema>;

const currencies = [
    { code: 'INR', name: 'Indian Rupee (₹)' },
    { code: 'USD', name: 'US Dollar ($)' },
    { code: 'EUR', name: 'Euro (€)' },
    { code: 'GBP', name: 'British Pound (£)' },
];

interface ColorFieldProps {
  form: any;
  name: keyof SettingsFormValues;
  label: string;
}

const ColorField: React.FC<ColorFieldProps> = ({ form, name, label }) => (
    <FormField
        control={form.control}
        name={name}
        render={({ field }) => (
            <FormItem>
            <FormLabel>{label}</FormLabel>
            <FormControl>
                <div className="relative">
                    <div className="absolute left-2 top-1/2 -translate-y-1/2 h-6 w-6 rounded-full border" style={{ backgroundColor: field.value }} />
                    <Input type="color" value={field.value} onChange={field.onChange} className="absolute inset-0 opacity-0 cursor-pointer" />
                    <Input readOnly value={field.value} className="pl-10" />
                </div>
            </FormControl>
            <FormMessage />
            </FormItem>
        )}
    />
);


export default function AdminSettingsPage() {
  const { toast } = useToast();
  const firestore = useFirestore();
  const settingsDocRef = useMemoFirebase(() => doc(firestore, 'settings', 'main'), [firestore]);

  const { data: settings, isLoading, error } = useDoc<SettingsData>(settingsDocRef);
  
  const form = useForm<SettingsFormValues>({
    resolver: zodResolver(settingsSchema),
    defaultValues: {
      websiteName: 'ChandraBrothers',
      websiteTagline: 'Your trusted online medical store.',
      logoUrl: '',
      logoIcon: 'Stethoscope',
      background: '#f5f5f5',
      foreground: '#0a0a0a',
      card: '#ffffff',
      primary: '#0078ff',
      primaryForeground: '#fafafa',
      secondary: '#00b4ac',
      secondaryForeground: '#fafafa',
      destructive: '#ff0000',
      destructiveForeground: '#fafafa',
      ring: '#0078ff',
      stripeApiKey: '',
      paypalClientId: '',
      razorpayApiKey: '',
      enableStripe: false,
      enablePaypal: false,
      enableRazorpay: false,
      defaultCurrency: 'INR',
      paymentTestMode: true,
      publicApiKey: '',
      ipWhitelist: '',
      googleAnalyticsId: '',
      facebookPixelId: '',
      googleTagManagerId: '',
      enableCache: true,
      imageCompression: 'medium',
      maintenanceMode: false,
      maintenancePageBanner: '',
      homepageSections: [
        { id: 'hero', name: 'Hero Section', enabled: true },
        { id: 'categories', name: 'Category Slider', enabled: true },
        { id: 'trending', name: 'Trending Products', enabled: true },
        { id: 'bestsellers', name: 'Best Sellers', enabled: true },
        { id: 'services', name: 'Service Strip', enabled: true },
        { id: 'newarrivals', name: 'New Arrivals', enabled: true },
      ],
      headerScripts: '',
      bodyStartScripts: '',
      footerScripts: '',
    },
  });
  
  useEffect(() => {
    if (settings) {
      form.reset(settings);
    }
  }, [settings, form]);
  
  const onSubmit = async (values: SettingsFormValues) => {
    try {
      await updateSettings(firestore, values);
      toast({
        title: 'Settings Saved',
        description: 'Your website settings have been updated.',
      });

    } catch (e) {
      // Error is handled by the global emitter in updateSettings
      toast({
        variant: 'destructive',
        title: 'Save Failed',
        description: 'Could not save settings. Please try again.',
      });
    }
  };

  const handleBackup = () => {
    toast({
      title: 'Backup Initiated (Demo)',
      description: 'A full backup of your website data is being generated. This is a demo feature.',
    });
  };

  const isSubmitting = form.formState.isSubmitting;
  const logoUrl = form.watch('logoUrl');

  if (isLoading) {
    return (
        <div className="p-4 md:p-8 space-y-6">
            <Skeleton className="h-8 w-64" />
            <Card>
                <CardHeader>
                    <Skeleton className="h-6 w-48" />
                    <Skeleton className="h-4 w-full max-w-sm mt-2" />
                </CardHeader>
                <CardContent className="space-y-6">
                    <div className="space-y-2">
                        <Skeleton className="h-4 w-24" />
                        <Skeleton className="h-10 w-full" />
                    </div>
                     <div className="space-y-2">
                        <Skeleton className="h-4 w-24" />
                        <Skeleton className="h-10 w-full" />
                    </div>
                </CardContent>
            </Card>
        </div>
    )
  }

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div>
        <h1 className="text-3xl font-bold">Website Settings</h1>
        <p className="text-muted-foreground">Manage your store's branding, payments, and global configuration.</p>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            <Card>
            <CardHeader>
                <CardTitle className="flex items-center gap-2">
                <Brush />
                Branding & Identity
                </CardTitle>
                <CardDescription>
                Customize your website's name, tagline, and logo.
                </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
                <FormField
                    control={form.control}
                    name="websiteName"
                    render={({ field }) => (
                        <FormItem>
                        <FormLabel>Website Name</FormLabel>
                        <FormControl>
                            <Input placeholder="Your Website Name" {...field} />
                        </FormControl>
                        <FormMessage />
                        </FormItem>
                    )}
                />
                <FormField
                    control={form.control}
                    name="websiteTagline"
                    render={({ field }) => (
                        <FormItem>
                        <FormLabel>Tagline</FormLabel>
                        <FormControl>
                            <Input placeholder="A short, catchy phrase for your store" {...field} />
                        </FormControl>
                         <FormMessage />
                        </FormItem>
                    )}
                />
                <Separator />
                <div>
                  <FormLabel>Logo</FormLabel>
                  <div className="mt-2 flex flex-col md:flex-row items-start gap-6">
                    <div className="relative h-20 w-20 flex-shrink-0 rounded-md border p-2 flex items-center justify-center bg-muted/50">
                      {logoUrl ? (
                        <Image src={logoUrl} alt="Website Logo" layout="fill" objectFit="contain" />
                      ) : (
                        <ImageIcon className="h-8 w-8 text-muted-foreground" />
                      )}
                    </div>
                    <div className="flex-grow space-y-4">
                        <FormField
                            control={form.control}
                            name="logoUrl"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Logo URL</FormLabel>
                                <FormControl>
                                    <Input type="url" placeholder="https://example.com/logo.png" {...field} />
                                </FormControl>
                                <FormDescription>Provide a direct link to your logo image.</FormDescription>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="logoIcon"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Fallback Icon Name</FormLabel>
                                <FormControl>
                                    <Input placeholder="e.g., Stethoscope" {...field} />
                                </FormControl>
                                <FormDescription>A Lucide icon name to use when the logo can't be shown.</FormDescription>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </div>
                  </div>
                </div>

            </CardContent>
            </Card>
            
             <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                        <Palette />
                        Theme Customizer
                    </CardTitle>
                    <CardDescription>Manage your global color palette.</CardDescription>
                </CardHeader>
                <CardContent className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
                    <ColorField form={form} name="background" label="Background" />
                    <ColorField form={form} name="foreground" label="Foreground" />
                    <ColorField form={form} name="card" label="Card" />
                    <ColorField form={form} name="primary" label="Primary" />
                    <ColorField form={form} name="primaryForeground" label="Primary FG" />
                    <ColorField form={form} name="secondary" label="Secondary" />
                    <ColorField form={form} name="secondaryForeground" label="Secondary FG" />
                    <ColorField form={form} name="destructive" label="Destructive" />
                    <ColorField form={form} name="destructiveForeground" label="Destructive FG" />
                    <ColorField form={form} name="ring" label="Ring" />
                </CardContent>
            </Card>


            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2"><CreditCard /> Payment Gateway Configuration</CardTitle>
                    <CardDescription>Manage keys, methods, and modes for your payment providers.</CardDescription>
                </CardHeader>
                 <CardContent className="space-y-6">
                    <div className="space-y-4 p-4 border rounded-lg">
                        <FormField
                            control={form.control}
                            name="enableStripe"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between">
                                <FormLabel className="font-bold">Enable Stripe</FormLabel>
                                <FormControl>
                                    <Switch checked={field.value} onCheckedChange={field.onChange} />
                                </FormControl>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="stripeApiKey"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Stripe API Key</FormLabel>
                                <FormControl>
                                    <Input type="password" placeholder="sk_test_..." {...field} />
                                </FormControl>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </div>
                     <div className="space-y-4 p-4 border rounded-lg">
                        <FormField
                            control={form.control}
                            name="enablePaypal"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between">
                                <FormLabel className="font-bold">Enable PayPal</FormLabel>
                                <FormControl>
                                    <Switch checked={field.value} onCheckedChange={field.onChange} />
                                </FormControl>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="paypalClientId"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>PayPal Client ID</FormLabel>
                                <FormControl>
                                    <Input type="password" placeholder="AZD..." {...field} />
                                </FormControl>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </div>
                    <div className="space-y-4 p-4 border rounded-lg">
                        <FormField
                            control={form.control}
                            name="enableRazorpay"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between">
                                <FormLabel className="font-bold">Enable Razorpay</FormLabel>
                                <FormControl>
                                    <Switch checked={field.value} onCheckedChange={field.onChange} />
                                </FormControl>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="razorpayApiKey"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Razorpay API Key</FormLabel>
                                <FormControl>
                                    <Input type="password" placeholder="rzp_test_..." {...field} />
                                </FormControl>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </div>
                    <Separator />
                     <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                          <FormField
                            control={form.control}
                            name="defaultCurrency"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Default Currency</FormLabel>
                                <Select onValueChange={field.onChange} defaultValue={field.value}>
                                    <FormControl>
                                    <SelectTrigger>
                                        <SelectValue placeholder="Select a currency" />
                                    </SelectTrigger>
                                    </FormControl>
                                    <SelectContent>
                                    {currencies.map(c => (
                                        <SelectItem key={c.code} value={c.code}>{c.name}</SelectItem>
                                    ))}
                                    </SelectContent>
                                </Select>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="paymentTestMode"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3 shadow-sm h-fit mt-auto">
                                    <div className="space-y-0.5">
                                        <FormLabel>Test Mode</FormLabel>
                                        <FormMessage />
                                    </div>
                                    <FormControl>
                                        <Switch checked={field.value} onCheckedChange={field.onChange} aria-label="Toggle Payment Test Mode" />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                     </div>
                 </CardContent>
            </Card>

            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2"><Shield /> API &amp; Security</CardTitle>
                    <CardDescription>Manage API access and security settings.</CardDescription>
                </CardHeader>
                 <CardContent className="space-y-6">
                    <FormField
                        control={form.control}
                        name="publicApiKey"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Public API Key</FormLabel>
                            <FormControl>
                                <Input placeholder="pub_..." {...field} />
                            </FormControl>
                            <FormDescription>This key can be used for client-side requests.</FormDescription>
                            <FormMessage />
                            </FormItem>
                        )}
                    />
                     <FormField
                        control={form.control}
                        name="ipWhitelist"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Admin IP Whitelist</FormLabel>
                            <FormControl>
                                <Textarea placeholder="e.g., 192.168.1.1, 2001:0db8::" {...field} />
                            </FormControl>
                            <FormDescription>Comma-separated list of IP addresses that can access the admin panel.</FormDescription>
                            <FormMessage />
                            </FormItem>
                        )}
                    />
                 </CardContent>
            </Card>

            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2"><BarChart /> Integrations</CardTitle>
                    <CardDescription>Manage third-party integrations and marketing analytics.</CardDescription>
                </CardHeader>
                 <CardContent className="space-y-6">
                    <FormField
                        control={form.control}
                        name="googleAnalyticsId"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Google Analytics ID</FormLabel>
                            <FormControl>
                                <Input placeholder="G-..." {...field} />
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                    />
                    <FormField
                        control={form.control}
                        name="facebookPixelId"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Facebook Pixel ID</FormLabel>
                            <FormControl>
                                <Input placeholder="e.g., 123456789012345" {...field} />
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                    />
                     <FormField
                        control={form.control}
                        name="googleTagManagerId"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Google Tag Manager ID</FormLabel>
                            <FormControl>
                                <Input placeholder="GTM-..." {...field} />
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                    />
                 </CardContent>
            </Card>

            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2"><Server /> Performance &amp; Optimization</CardTitle>
                    <CardDescription>Manage caching, image optimization, and other performance settings.</CardDescription>
                </CardHeader>
                <CardContent className="space-y-6">
                    <FormField
                        control={form.control}
                        name="enableCache"
                        render={({ field }) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3 shadow-sm">
                                <div className="space-y-0.5">
                                    <FormLabel>Enable Website Cache</FormLabel>
                                    <FormDescription>
                                        Improves load times by caching static assets.
                                    </FormDescription>
                                </div>
                                <FormControl>
                                    <Switch
                                        checked={field.value}
                                        onCheckedChange={field.onChange}
                                    />
                                </FormControl>
                            </FormItem>
                        )}
                    />
                     <FormField
                        control={form.control}
                        name="imageCompression"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Image Compression Level</FormLabel>
                            <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                <SelectTrigger>
                                    <SelectValue placeholder="Select compression level" />
                                </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                    <SelectItem value="low">Low (Higher Quality)</SelectItem>
                                    <SelectItem value="medium">Medium (Balanced)</SelectItem>
                                    <SelectItem value="high">High (Faster Loading)</SelectItem>
                                </SelectContent>
                            </Select>
                            <FormDescription>
                                Balances image quality and file size for faster page loads.
                            </FormDescription>
                            <FormMessage />
                            </FormItem>
                        )}
                        />
                </CardContent>
            </Card>
            
            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2"><LayoutTemplate /> Homepage Layout Manager</CardTitle>
                    <CardDescription>Drag and drop to reorder sections on your homepage.</CardDescription>
                </CardHeader>
                <CardContent>
                    <HomepageSectionsManager form={form} />
                </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Code /> Custom HTML & Scripts
                </CardTitle>
                <CardDescription>
                  Insert custom code snippets into your site's header or footer. Use with caution.
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                 <FormField
                    control={form.control}
                    name="headerScripts"
                    render={({ field }) => (
                        <FormItem>
                            <FormLabel>Header Scripts</FormLabel>
                             <FormControl>
                                <Textarea
                                placeholder="e.g., <meta ...>, <link ...>, <script ...>"
                                className="font-mono min-h-32"
                                {...field}
                                />
                            </FormControl>
                            <FormDescription>
                                Injects code before the closing `&lt;/head&gt;` tag.
                            </FormDescription>
                            <FormMessage />
                        </FormItem>
                    )}
                 />
                 <FormField
                    control={form.control}
                    name="bodyStartScripts"
                    render={({ field }) => (
                        <FormItem>
                            <FormLabel>Body Start Scripts</FormLabel>
                             <FormControl>
                                <Textarea
                                placeholder="e.g., GTM noscript iframe"
                                className="font-mono min-h-24"
                                {...field}
                                />
                            </FormControl>
                            <FormDescription>
                                Injects code right after the opening `&lt;body&gt;` tag.
                            </FormDescription>
                            <FormMessage />
                        </FormItem>
                    )}
                 />
                 <FormField
                    control={form.control}
                    name="footerScripts"
                    render={({ field }) => (
                        <FormItem>
                            <FormLabel>Footer Scripts</FormLabel>
                             <FormControl>
                                <Textarea
                                placeholder="e.g., Chat widgets, tracking scripts"
                                className="font-mono min-h-32"
                                {...field}
                                />
                            </FormControl>
                            <FormDescription>
                                Injects code before the closing `&lt;/body&gt;` tag.
                            </FormDescription>
                            <FormMessage />
                        </FormItem>
                    )}
                 />
              </CardContent>
            </Card>

            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2"><Download /> Backup &amp; Restore</CardTitle>
                    <CardDescription>Manage website data backups and restore points.</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                    <p className="text-sm text-muted-foreground">
                        Create a manual backup of your website data, including your Firestore database and configuration settings.
                    </p>
                    <Button variant="outline" type="button" onClick={handleBackup}>
                        <Download className="mr-2" />
                        Download Full Backup
                    </Button>
                </CardContent>
            </Card>

            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2"><HardHat /> Maintenance &amp; DevOps</CardTitle>
                    <CardDescription>Control maintenance mode, view logs, and manage deployments.</CardDescription>
                </CardHeader>
                <CardContent className="space-y-6">
                    <FormField
                        control={form.control}
                        name="maintenanceMode"
                        render={({ field }) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3 shadow-sm">
                                <div className="space-y-0.5">
                                    <FormLabel>Enable Maintenance Mode</FormLabel>
                                    <FormDescription>
                                        Redirect all traffic to a maintenance page.
                                    </FormDescription>
                                </div>
                                <FormControl>
                                    <Switch
                                        checked={field.value}
                                        onCheckedChange={field.onChange}
                                    />
                                </FormControl>
                            </FormItem>
                        )}
                    />
                     <FormField
                        control={form.control}
                        name="maintenancePageBanner"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel>Maintenance Banner Message</FormLabel>
                                <FormControl>
                                    <Input placeholder="e.g., We'll be back shortly!" {...field} />
                                </FormControl>
                                <FormDescription>
                                    This message will be displayed on the maintenance page.
                                </FormDescription>
                                <FormMessage />
                            </FormItem>
                        )}
                    />
                </CardContent>
            </Card>

            <div className="flex justify-end">
                <Button type="submit" disabled={isSubmitting}>
                    {isSubmitting && <LoaderCircle className="animate-spin mr-2" />}
                    Save All Changes
                </Button>
            </div>
        </form>
      </Form>
    </div>
  );
}
    
