
'use client';

import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Button } from '@/components/ui/button';
import { Users, Search, MoreHorizontal, AlertTriangle, View, Trash2 } from 'lucide-react';
import { Input } from '@/components/ui/input';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection } from 'firebase/firestore';
import type { User, Admin } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import Link from 'next/link';

export default function AdminUsersPage() {
  const firestore = useFirestore();
  
  const usersCollectionRef = useMemoFirebase(
    () => collection(firestore, 'users'),
    [firestore]
  );
  const adminRolesCollectionRef = useMemoFirebase(
    () => collection(firestore, 'roles_admin'),
    [firestore]
  );

  const { data: users, isLoading: usersLoading, error: usersError } = useCollection<User>(usersCollectionRef);
  const { data: admins, isLoading: adminsLoading, error: adminsError } = useCollection<Admin>(adminRolesCollectionRef);

  const isLoading = usersLoading || adminsLoading;
  const error = usersError || adminsError;

  const nonAdminUsers = useMemoFirebase(() => {
    if (!users || !admins) return [];
    const adminIds = new Set(admins.map(admin => admin.id));
    return users.filter(user => !adminIds.has(user.id));
  }, [users, admins]);

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div className="flex flex-col md:flex-row justify-between md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold">User Management</h1>
          <p className="text-muted-foreground">
            View, search, and manage all registered non-admin users.
          </p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>All Users</CardTitle>
          <CardDescription>
            A complete list of all non-admin users on the platform.
          </CardDescription>
          <div className="relative pt-2">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
            <Input
              placeholder="Search by user name or email..."
              className="pl-10 text-base"
            />
          </div>
        </CardHeader>
        <CardContent>
          {isLoading && (
             <div className="space-y-2">
                <Skeleton className="h-12 w-full" />
                <Skeleton className="h-12 w-full" />
                <Skeleton className="h-12 w-full" />
            </div>
          )}
          {error && (
             <Alert variant="destructive">
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle>Error Loading Users</AlertTitle>
                <AlertDescription>
                  Could not fetch user data. Please ensure you have the correct permissions.
                </AlertDescription>
              </Alert>
          )}
          {!isLoading && !error && nonAdminUsers && nonAdminUsers.length > 0 ? (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Name</TableHead>
                  <TableHead>Email</TableHead>
                  <TableHead>Registered On</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {nonAdminUsers.map(user => (
                  <TableRow key={user.id}>
                    <TableCell className="font-medium">{user.name}</TableCell>
                    <TableCell>{user.email}</TableCell>
                    <TableCell>{user.registeredDate ? new Date(user.registeredDate.seconds * 1000).toLocaleDateString() : 'N/A'}</TableCell>
                    <TableCell className="text-right">
                       <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" size="icon">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem asChild>
                              <Link href={`/admin/users/${user.id}`}>
                                <View className="mr-2 h-4 w-4" />
                                View Details
                              </Link>
                            </DropdownMenuItem>
                             <DropdownMenuItem className="text-destructive">
                              <Trash2 className="mr-2 h-4 w-4" />
                              Delete
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          ) : (
            !isLoading && !error && (
                <Alert>
                <Users className="h-4 w-4" />
                <AlertTitle>No Users Found</AlertTitle>
                <AlertDescription>
                    There are no non-admin users to display at the moment.
                </AlertDescription>
                </Alert>
            )
          )}
        </CardContent>
      </Card>
    </div>
  );
}

