
'use client';

import React from 'react';
import { Header } from '@/components/layout/header';
import { Footer } from '@/components/layout/footer';
import { Card, CardContent, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { AlertTriangle, LoaderCircle, ShoppingCart } from 'lucide-react';
import Link from 'next/link';
import { Button } from '@/components/ui/button';
import { useUser, useFirestore, useCollection, useMemoFirebase } from '@/firebase';
import { collection, query, where, getDocs } from 'firebase/firestore';
import type { CartItem, Product } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { CartItemCard } from '@/components/cart-item-card';
import { Separator } from '@/components/ui/separator';

function CartContents() {
    const { user } = useUser();
    const firestore = useFirestore();
    const [products, setProducts] = React.useState<Product[]>([]);
    const [productsLoading, setProductsLoading] = React.useState(true);
    
    const cartCollectionRef = useMemoFirebase(
        () => (user ? collection(firestore, 'users', user.uid, 'cart') : null),
        [firestore, user]
    );

    const {
        data: cartItemsData,
        isLoading: cartIsLoading,
        error: cartError,
    } = useCollection<CartItem>(cartCollectionRef);

    const cartItems = React.useMemo(() => cartItemsData, [cartItemsData]);

    React.useEffect(() => {
        const fetchProducts = async () => {
            if (!cartItems || cartItems.length === 0) {
                setProducts([]);
                setProductsLoading(false);
                return;
            }

            const productIds = [...new Set(cartItems.map(item => item.productId))];
            
            if (productIds.length === 0) {
              setProducts([]);
              setProductsLoading(false);
              return;
            }

            // In a real app, you would fetch from a `products` collection.
            const mockProducts: Product[] = [
                { id: '1', name: 'Gentle Baby Lotion', price: 12.99, imageId: 'prod-1', category: 'Baby Care' },
                { id: '2', name: "Men's Revitalizing Face Wash", price: 8.50, imageId: 'prod-2', category: "Men's Care" },
                { id: '3', name: 'Vitamin C Plus Zinc', price: 19.99, imageId: 'prod-3', category: 'Supplements' },
                { id: '4', name: 'Paracetamol 500mg Tablets', price: 2.99, imageId: 'prod-4', category: 'Medicines' },
                { id: '5', name: 'Precision Digital Thermometer', price: 25.00, imageId: 'prod-5', category: 'Equipment' },
                { id: '6', name: 'Ultra-thin Sanitary Pads', price: 5.49, imageId: 'prod-6', category: "Women's Care" },
                { id: '7', name: 'Whey Protein Isolate', price: 45.99, imageId: 'prod-7', category: 'Supplements' },
                { id: '8', name: 'Prefilled Insulin Syringe', price: 32.75, imageId: 'prod-8', category: 'Injections' },
            ];

            const fetchedProducts = mockProducts.filter(p => productIds.includes(p.id));
            setProducts(fetchedProducts);
            setProductsLoading(false);
        };

        fetchProducts();
    }, [cartItems]);

    const isLoading = cartIsLoading || productsLoading;
    const error = cartError;

    if (isLoading) {
        return <div className="space-y-4">
            <Skeleton className="h-28 w-full" />
            <Skeleton className="h-28 w-full" />
        </div>
    }

    if (error) {
        return (
            <div className="flex flex-col items-center justify-center text-center text-destructive bg-destructive/10 p-6 rounded-lg">
                <AlertTriangle className="h-8 w-8 mb-2" />
                <p className="font-semibold">Failed to load cart.</p>
                <p className="text-sm">Please check your connection or try again later.</p>
            </div>
        );
    }
    
    if (!cartItems || cartItems.length === 0) {
        return (
            <div className="flex flex-1 items-center justify-center">
                <div className="flex flex-col items-center justify-center rounded-lg border-2 border-dashed border-muted-foreground/30 bg-background p-12 text-center">
                  <div className="flex h-16 w-16 items-center justify-center rounded-full bg-primary/10 text-primary mb-4">
                    <ShoppingCart className="h-8 w-8" />
                  </div>
                  <h2 className="text-2xl font-semibold">Your cart is empty</h2>
                  <p className="mt-2 text-muted-foreground">
                    Looks like you haven't added anything to your cart yet.
                  </p>
                  <Button asChild className="mt-6">
                    <Link href="/">Start Shopping</Link>
                  </Button>
                </div>
            </div>
        )
    }

    const subtotal = cartItems.reduce((acc, item) => {
        const product = products.find(p => p.id === item.productId);
        return acc + (product ? product.price * item.quantity : 0);
    }, 0);
    const deliveryFee = subtotal > 50 ? 0 : 5.99;
    const total = subtotal + deliveryFee;

    return (
        <div className="container py-8 md:py-12">
          <h1 className="text-3xl font-bold mb-6">Shopping Cart</h1>
            <div className="grid md:grid-cols-3 gap-8 items-start">
                <Card className="md:col-span-2">
                    <CardHeader>
                        <CardTitle>Cart Items ({cartItems.length})</CardTitle>
                    </CardHeader>
                    <CardContent className="divide-y">
                       {cartItems.map(item => {
                           const product = products.find(p => p.id === item.productId);
                           if (!product) return null;
                           return <CartItemCard key={item.id} item={item} product={product} />
                       })}
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader>
                        <CardTitle>Order Summary</CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3 text-sm">
                        <div className="flex justify-between">
                            <span>Subtotal</span>
                            <span>₹{subtotal.toFixed(2)}</span>
                        </div>
                        <div className="flex justify-between text-muted-foreground">
                            <span>Delivery Fee</span>
                            <span>{deliveryFee === 0 ? 'Free' : `₹${deliveryFee.toFixed(2)}`}</span>
                        </div>
                         <Separator />
                         <div className="flex justify-between font-bold text-base">
                            <span>Total</span>
                            <span>₹{total.toFixed(2)}</span>
                        </div>
                    </CardContent>
                    <CardFooter>
                        <Button className="w-full">Proceed to Checkout</Button>
                    </CardFooter>
                </Card>
            </div>
        </div>
    )
}


export default function CartPage() {
  return (
    <div className="flex min-h-screen flex-col">
      <Header />
      <main className="flex flex-1 flex-col bg-muted/40">
        <CartContents />
      </main>
      <Footer />
    </div>
  );
}
