'use client';

import React, { useEffect, useState, useRef } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import {
  useUser,
  useFirestore,
  useDoc,
  useMemoFirebase,
} from '@/firebase';
import { doc } from 'firebase/firestore';
import { Skeleton } from '@/components/ui/skeleton';
import { User as UserIcon, Mail, Phone, Calendar, Edit, ShieldCheck, Camera } from 'lucide-react';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogFooter,
  DialogClose,
} from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import { useToast } from '@/hooks/use-toast';
import { setUserData } from '@/firebase/firestore/set-user-data';
import { updateProfile } from 'firebase/auth';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';

const profileFormSchema = z.object({
  name: z.string().min(2, 'Name must be at least 2 characters.'),
  phone: z.string().optional(),
});

type ProfileFormValues = z.infer<typeof profileFormSchema>;

export default function AccountPage() {
  const { user, isUserLoading } = useUser();
  const firestore = useFirestore();
  const { toast } = useToast();
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const userDocRef = useMemoFirebase(
    () => (user ? doc(firestore, 'users', user.uid) : null),
    [firestore, user]
  );

  const {
    data: userData,
    isLoading: isUserDataLoading,
    error,
  } = useDoc(userDocRef);

  const form = useForm<ProfileFormValues>({
    resolver: zodResolver(profileFormSchema),
    defaultValues: {
      name: '',
      phone: '',
    },
  });

  useEffect(() => {
    if (userData) {
      form.reset({
        name: userData.name || '',
        phone: userData.phone || '',
      });
    }
  }, [userData, user, form]);

  const registeredDate = userData?.registeredDate
    ? new Date(userData.registeredDate.seconds * 1000).toLocaleDateString()
    : 'N/A';
  
  const getInitials = (name: string) => {
    return name?.split(' ').map(n => n[0]).join('').toUpperCase() || '';
  }

  const handleAvatarClick = () => {
    fileInputRef.current?.click();
  }

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // TODO: Implement file upload to Firebase Storage
      toast({
        title: 'Photo Selected',
        description: `${file.name} is ready to be uploaded.`,
      });
    }
  };


  const onSubmit = async (values: ProfileFormValues) => {
    if (!user) return;
    setIsSubmitting(true);

    try {
      // Update Firebase Auth profile
      await updateProfile(user, { 
          displayName: values.name,
      });

      // Update Firestore document
      const updatedData = {
        name: values.name,
        phone: values.phone,
      };
      setUserData(firestore, user.uid, updatedData);

      toast({
        title: 'Profile Updated',
        description: 'Your profile has been successfully updated.',
      });
      setIsModalOpen(false);
    } catch (e: any) {
      toast({
        variant: 'destructive',
        title: 'Update Failed',
        description: e.message || 'Could not update your profile.',
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  const isLoading = isUserLoading || isUserDataLoading;

  return (
    <div className="p-4 md:p-8">
      <h1 className="text-2xl font-bold mb-6">My Account</h1>
      <Card>
        <CardHeader className="flex flex-row items-center justify-between border-b">
          <CardTitle>Profile Details</CardTitle>
          <Dialog open={isModalOpen} onOpenChange={setIsModalOpen}>
            <DialogTrigger asChild>
              <Button variant="outline" size="sm">
                <Edit className="mr-2 h-4 w-4" />
                Edit Profile
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Edit Profile</DialogTitle>
              </DialogHeader>
                <form
                  onSubmit={form.handleSubmit(onSubmit)}
                  className="space-y-4"
                >
                  <div className="flex flex-col items-center gap-4">
                     <div className="relative group cursor-pointer" onClick={handleAvatarClick}>
                        <Avatar className="h-24 w-24 text-3xl">
                          <AvatarImage src={user?.photoURL || undefined} alt={userData?.name} />
                          <AvatarFallback>{getInitials(userData?.name || '')}</AvatarFallback>
                        </Avatar>
                        <div className="absolute inset-0 bg-black/40 rounded-full flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity">
                           <Camera className="h-8 w-8 text-white" />
                        </div>
                        <input
                            type="file"
                            ref={fileInputRef}
                            onChange={handleFileChange}
                            className="hidden"
                            accept="image/png, image/jpeg"
                        />
                     </div>
                    <div>
                      <Label htmlFor="email">Email</Label>
                      <Input id="email" value={userData?.email || ''} disabled />
                    </div>
                  </div>
                  <div>
                    <Label htmlFor="name">Name</Label>
                    <Input id="name" {...form.register('name')} />
                    {form.formState.errors.name && (
                      <p className="text-destructive text-sm mt-1">
                        {form.formState.errors.name.message}
                      </p>
                    )}
                  </div>
                  <div>
                    <Label htmlFor="phone">Phone Number</Label>
                    <Input id="phone" {...form.register('phone')} />
                  </div>
                  <DialogFooter>
                    <DialogClose asChild>
                      <Button type="button" variant="ghost">
                        Cancel
                      </Button>
                    </DialogClose>
                    <Button type="submit" disabled={isSubmitting}>
                      {isSubmitting ? 'Saving...' : 'Save Changes'}
                    </Button>
                  </DialogFooter>
                </form>
            </DialogContent>
          </Dialog>
        </CardHeader>
        <CardContent className="p-6">
          {isLoading ? (
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div className="md:col-span-1 flex flex-col items-center md:items-start gap-4">
                <Skeleton className="h-24 w-24 rounded-full" />
                <Skeleton className="h-6 w-3/4" />
              </div>
              <div className="md:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-y-4 gap-x-6">
                <div className="space-y-1"><Skeleton className="h-4 w-24" /><Skeleton className="h-5 w-full" /></div>
                <div className="space-y-1"><Skeleton className="h-4 w-24" /><Skeleton className="h-5 w-full" /></div>
                <div className="space-y-1"><Skeleton className="h-4 w-24" /><Skeleton className="h-5 w-full" /></div>
                <div className="space-y-1"><Skeleton className="h-4 w-24" /><Skeleton className="h-5 w-full" /></div>
              </div>
            </div>
          ) : error ? (
            <p className="text-destructive">
              Could not load profile data.
            </p>
          ) : userData ? (
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
               <div className="md:col-span-1 flex flex-col items-center md:items-start gap-4">
                  <Avatar className="h-24 w-24 text-3xl">
                      <AvatarImage src={user?.photoURL || undefined} alt={userData.name} />
                      <AvatarFallback>{getInitials(userData.name)}</AvatarFallback>
                  </Avatar>
                  <div className="text-center md:text-left">
                      <p className="text-xl font-semibold">{userData.name}</p>
                      <p className="text-sm text-muted-foreground">{userData.email}</p>
                  </div>
              </div>

               <div className="md:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-y-6 gap-x-8 pt-2">
                  <div>
                    <Label className="text-xs text-muted-foreground">Phone Number</Label>
                    <p className="text-sm font-medium flex items-center gap-2 mt-1">
                      <Phone className="w-4 h-4 text-primary" />
                      {userData.phone || 'Not provided'}
                    </p>
                  </div>
                  <div>
                    <Label className="text-xs text-muted-foreground">Registered On</Label>
                    <p className="text-sm font-medium flex items-center gap-2 mt-1">
                      <Calendar className="w-4 h-4 text-primary" />
                      {registeredDate}
                    </p>
                  </div>
                  <div>
                    <Label className="text-xs text-muted-foreground">Email Address</Label>
                    <p className="text-sm font-medium flex items-center gap-2 mt-1">
                       <Mail className="w-4 h-4 text-primary" />
                       {userData.email} <ShieldCheck className="w-4 h-4 text-green-500" />
                    </p>
                  </div>
               </div>
            </div>
          ) : (
            <p>No profile data found.</p>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
