'use client';

import React, { useState, useMemo } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { PlusCircle, Home, MapPin, LoaderCircle, AlertTriangle } from 'lucide-react';
import {
  useUser,
  useFirestore,
  useCollection,
  useMemoFirebase,
} from '@/firebase';
import { collection } from 'firebase/firestore';
import type { Address } from '@/lib/data';
import { AddressCard } from '@/components/address-card';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { AddressForm } from '@/components/address-form';
import { Skeleton } from '@/components/ui/skeleton';

export default function AddressPage() {
  const { user } = useUser();
  const firestore = useFirestore();
  const [isModalOpen, setIsModalOpen] = useState(false);

  const addressesCollectionRef = useMemoFirebase(
    () =>
      user ? collection(firestore, 'users', user.uid, 'addresses') : null,
    [firestore, user]
  );

  const {
    data: addresses,
    isLoading,
    error,
  } = useCollection<Address>(addressesCollectionRef);

  return (
    <div className="p-4 md:p-8">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold">My Addresses</h1>
        <Dialog open={isModalOpen} onOpenChange={setIsModalOpen}>
          <DialogTrigger asChild>
            <Button>
              <PlusCircle className="mr-2 h-4 w-4" />
              Add New Address
            </Button>
          </DialogTrigger>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Add a New Address</DialogTitle>
            </DialogHeader>
            <AddressForm
              onSave={() => setIsModalOpen(false)}
              onCancel={() => setIsModalOpen(false)}
            />
          </DialogContent>
        </Dialog>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <MapPin className="h-5 w-5 text-primary" />
            Saved Addresses
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading && (
            <div className="space-y-4">
              <Skeleton className="h-24 w-full rounded-lg" />
              <Skeleton className="h-24 w-full rounded-lg" />
            </div>
          )}

          {error && (
             <div className="flex flex-col items-center justify-center text-center text-destructive bg-destructive/10 p-6 rounded-lg">
                <AlertTriangle className="h-8 w-8 mb-2" />
                <p className="font-semibold">Failed to load addresses.</p>
                <p className="text-sm">Please check your connection or try again later.</p>
            </div>
          )}

          {!isLoading && !error && addresses && addresses.length > 0 && (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {addresses.map((address) => (
                <AddressCard key={address.id} address={address} />
              ))}
            </div>
          )}

          {!isLoading && !error && (!addresses || addresses.length === 0) && (
            <div className="flex flex-col items-center justify-center text-center text-muted-foreground bg-muted/40 p-10 rounded-lg">
              <Home className="h-10 w-10 mb-4" />
              <p className="font-semibold">No Saved Addresses</p>
              <p className="text-sm">
                You haven't added any addresses yet. Add one to get started!
              </p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
