'use client';

import React, { useState } from 'react';
import { useRouter } from 'next/navigation';
import {
  Card,
  CardHeader,
  CardTitle,
  CardContent,
  CardDescription,
  CardFooter,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from '@/components/ui/alert-dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { useAuth, useUser } from '@/firebase';
import { EmailAuthProvider, reauthenticateWithCredential, deleteUser } from 'firebase/auth';
import { useToast } from '@/hooks/use-toast';
import { LoaderCircle, ShieldAlert, Eye, EyeOff } from 'lucide-react';

export default function DeleteAccountPage() {
  const { user } = useUser();
  const auth = useAuth();
  const router = useRouter();
  const { toast } = useToast();

  const [password, setPassword] = useState('');
  const [isDeleting, setIsDeleting] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [showPassword, setShowPassword] = useState(false);

  const handleDeleteAccount = async () => {
    if (!user || !user.email) {
      toast({
        variant: 'destructive',
        title: 'Error',
        description: 'No user is currently logged in.',
      });
      return;
    }
    
    if (!password) {
        setError('Password is required to delete your account.');
        return;
    }

    setIsDeleting(true);
    setError(null);

    try {
      // Re-authenticate the user to confirm their identity
      const credential = EmailAuthProvider.credential(user.email, password);
      await reauthenticateWithCredential(user, credential);

      // If re-authentication is successful, delete the user
      await deleteUser(user);

      toast({
        title: 'Account Deleted',
        description: 'Your account has been permanently deleted.',
      });

      // Redirect to homepage after deletion
      router.push('/');

    } catch (error: any) {
      let errorMessage = 'An unexpected error occurred. Please try again.';
      if (error.code === 'auth/wrong-password') {
        errorMessage = 'The password you entered is incorrect. Please try again.';
      } else if (error.code === 'auth/requires-recent-login') {
        errorMessage = 'This is a sensitive operation. Please log in again before deleting your account.';
      }
      setError(errorMessage);
      setIsDeleting(false);
    }
  };

  return (
    <div className="p-4 md:p-8">
      <h1 className="text-2xl font-bold mb-6">Delete Account</h1>
      <Card className="max-w-2xl border-destructive/50">
        <CardHeader>
          <CardTitle className="flex items-center gap-2 text-destructive">
            <ShieldAlert className="h-6 w-6" />
            Danger Zone
          </CardTitle>
          <CardDescription>
            This action is permanent and cannot be undone. Please be certain before proceeding.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <p>
            Deleting your account will remove all of your personal information, saved addresses,
            and order history. This data will not be recoverable.
          </p>
        </CardContent>
        <CardFooter className="bg-destructive/5 p-4 flex justify-end">
          <AlertDialog onOpenChange={() => {
            // Reset state when dialog is closed or opened
            setPassword('');
            setError(null);
            setIsDeleting(false);
            setShowPassword(false);
          }}>
            <AlertDialogTrigger asChild>
              <Button variant="destructive">Delete My Account</Button>
            </AlertDialogTrigger>
            <AlertDialogContent>
              <AlertDialogHeader>
                <AlertDialogTitle>Are you absolutely sure?</AlertDialogTitle>
                <AlertDialogDescription>
                  This action is permanent. To confirm, please enter your password below. Your account and all associated data will be deleted.
                </AlertDialogDescription>
              </AlertDialogHeader>
              <div className="py-4 space-y-2">
                <Label htmlFor="password">Password</Label>
                <div className="relative">
                    <Input
                      id="password"
                      type={showPassword ? 'text' : 'password'}
                      value={password}
                      onChange={(e) => setPassword(e.target.value)}
                      placeholder="Enter your password..."
                      disabled={isDeleting}
                    />
                    <Button
                      type="button"
                      variant="ghost"
                      size="icon"
                      className="absolute right-1 top-1/2 h-7 w-7 -translate-y-1/2 text-muted-foreground"
                      onClick={() => setShowPassword((prev) => !prev)}
                    >
                      {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                      <span className="sr-only">
                        {showPassword ? 'Hide password' : 'Show password'}
                      </span>
                    </Button>
                </div>
                {error && <p className="text-sm text-destructive">{error}</p>}
              </div>
              <AlertDialogFooter>
                <AlertDialogCancel disabled={isDeleting}>Cancel</AlertDialogCancel>
                <AlertDialogAction onClick={handleDeleteAccount} disabled={!password || isDeleting} className="bg-destructive hover:bg-destructive/90">
                  {isDeleting && <LoaderCircle className="mr-2 h-4 w-4 animate-spin" />}
                  {isDeleting ? 'Deleting...' : 'Delete Account'}
                </AlertDialogAction>
              </AlertDialogFooter>
            </AlertDialogContent>
          </AlertDialog>
        </CardFooter>
      </Card>
    </div>
  );
}
