
'use client';

import { useState } from 'react';
import { Card, CardHeader, CardTitle, CardContent, CardDescription } from '@/components/ui/card';
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from '@/components/ui/accordion';
import { Input } from '@/components/ui/input';
import {
  LifeBuoy,
  Mail,
  Phone,
  Search,
  Truck,
  MessageSquare,
  FileText,
  Ticket,
  BookOpen,
  CreditCard,
  Package,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import Link from 'next/link';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { SupportTicketForm } from '@/components/support-ticket-form';


const quickActions = [
  { label: 'Track My Order', icon: Truck, href: '/dashboard/orders' },
  { label: 'Create Support Ticket', icon: Ticket, href: '#', isDialog: true },
  { label: 'Track Support Tickets', icon: FileText, href: '/dashboard/tickets' },
  { label: 'View FAQs', icon: BookOpen, href: '#faq' },
  { label: 'Chat with Us', icon: MessageSquare, href: '#' },
];

const helpCategories = [
  {
    title: 'Order Issues',
    description: 'Track, cancel, or get help with your order.',
    icon: Package,
    href: '#',
  },
  {
    title: 'Payments & Refunds',
    description: 'Manage payments, view invoices, and request refunds.',
    icon: CreditCard,
    href: '#',
  },
  {
    title: 'Prescriptions',
    description: 'Upload and manage your medical prescriptions.',
    icon: FileText,
    href: '#',
  },
  {
    title: 'Contact Support',
    description: 'Reach out to our team for direct assistance.',
    icon: Phone,
    href: '#contact',
  },
];

const faqs = [
    {
        question: "How do I place an order?",
        answer: "To place an order, simply browse our products, add items to your cart, and proceed to checkout. You will be asked to provide shipping and payment information."
    },
    {
        question: "How can I track my order?",
        answer: "You can track your order status in the 'Track Orders' section of your dashboard. Once your order is shipped, you will receive a tracking ID."
    },
    {
        question: "What is the return policy?",
        answer: "We accept returns within 14 days of delivery for most items. Please check our detailed Return & Refund Policy in the 'Terms & Conditions' section for more information."
    },
    {
        question: "How do I upload a prescription?",
        answer: "For medicines that require a prescription, you will see an option to upload it during the checkout process. Ensure the image is clear and legible."
    }
]

export default function HelpPage() {
    const [isTicketModalOpen, setIsTicketModalOpen] = useState(false);

  return (
    <div className="p-4 md:p-8 space-y-8">
      <Dialog open={isTicketModalOpen} onOpenChange={setIsTicketModalOpen}>
      <div className="text-center">
        <h1 className="text-3xl font-bold mb-2">Help Centre</h1>
        <p className="text-muted-foreground">
          How can we help you today?
        </p>
      </div>

      {/* Search Bar */}
      <div className="max-w-2xl mx-auto">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
          <Input placeholder="Search for topics, questions..." className="pl-10 h-12 text-base" />
        </div>
      </div>

      {/* Quick Actions */}
       <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4">
        {quickActions.map((action) => (
          action.isDialog ? (
            <DialogTrigger asChild key={action.label}>
                <Card className="text-center p-4 hover:bg-muted/50 hover:border-primary transition-all flex flex-col items-center justify-center h-full cursor-pointer">
                    <action.icon className="h-8 w-8 text-primary mb-2" />
                    <p className="font-semibold text-sm">{action.label}</p>
                </Card>
            </DialogTrigger>
          ) : (
            <Link href={action.href} key={action.label}>
                <Card className="text-center p-4 hover:bg-muted/50 hover:border-primary transition-all flex flex-col items-center justify-center h-full">
                  <action.icon className="h-8 w-8 text-primary mb-2" />
                  <p className="font-semibold text-sm">{action.label}</p>
                </Card>
            </Link>
          )
        ))}
      </div>


      {/* Help Categories */}
      <div>
         <h2 className="text-2xl font-bold mb-4">Browse by Category</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {helpCategories.map((category) => (
            <Link href={category.href} key={category.title}>
                <Card className="p-6 text-center flex flex-col items-center hover:shadow-lg transition-shadow">
                  <category.icon className="h-10 w-10 text-primary mb-3" />
                  <h3 className="font-semibold text-lg">{category.title}</h3>
                  <p className="text-sm text-muted-foreground mt-1">{category.description}</p>
                </Card>
            </Link>
          ))}
        </div>
      </div>

      {/* FAQ Section */}
      <Card id="faq">
          <CardHeader>
              <CardTitle>Frequently Asked Questions</CardTitle>
          </CardHeader>
          <CardContent>
              <Accordion type="single" collapsible className="w-full">
                  {faqs.map((faq, index) => (
                    <AccordionItem key={index} value={`item-${index + 1}`}>
                        <AccordionTrigger>{faq.question}</AccordionTrigger>
                        <AccordionContent>
                           {faq.answer}
                        </AccordionContent>
                    </AccordionItem>
                  ))}
              </Accordion>
          </CardContent>
      </Card>

      {/* Contact Section */}
      <Card id="contact">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <LifeBuoy className="h-5 w-5 text-primary" />
            Still Need Help?
          </CardTitle>
          <CardDescription>
            If you can't find the answer you're looking for, please don't hesitate to reach out to our support team.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col sm:flex-row gap-4">
            <DialogTrigger asChild>
                <Button variant="outline">
                    <Mail className="mr-2 h-4 w-4" /> Email Support
                </Button>
            </DialogTrigger>
            <Button asChild variant="outline">
              <a href="tel:+1234567890">
                <Phone className="mr-2 h-4 w-4" /> Call Us
              </a>
            </Button>
             <Button variant="outline">
                <MessageSquare className="mr-2 h-4 w-4" /> Live Chat
            </Button>
          </div>
        </CardContent>
      </Card>

      <DialogContent className="sm:max-w-[600px]">
        <DialogHeader>
          <DialogTitle>Create a Support Ticket</DialogTitle>
          <CardDescription>
            Please fill out the form below and we will get back to you as soon as possible.
          </CardDescription>
        </DialogHeader>
        <SupportTicketForm onSave={() => setIsTicketModalOpen(false)} />
      </DialogContent>
    </Dialog>
    </div>
  );
}
