
'use client';

import { useUser } from '@/firebase';
import {
  Card,
  CardHeader,
  CardTitle,
  CardContent,
} from '@/components/ui/card';
import {
  Clock,
  XCircle,
  Heart,
  ShoppingCart,
  Bell,
  Info,
} from 'lucide-react';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogDescription,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import React from 'react';
import Link from 'next/link';

const initialNotifications = [
  {
    id: 1,
    title: 'Order Shipped!',
    message: 'Your order #MED456-1 containing Paracetamol 500mg has been shipped and is on its way. You can track it in the orders section.',
    date: '2 hours ago',
    read: false,
    cta: {
      href: '/dashboard/orders/MED456-1',
      text: 'Track Order'
    }
  },
  {
    id: 2,
    title: 'Special Offer Just for You',
    message: 'Don\'t miss out! Get a special 20% discount on all supplements for a limited time. Stock up on your favorite wellness products now.',
    date: '1 day ago',
    read: false,
    cta: {
      href: '#',
      text: 'Shop Now'
    }
  },
  {
    id: 3,
    title: 'Order Delivered',
    message: 'Your order #MED123-3 has been successfully delivered. We hope you are satisfied with your purchase. Thank you for shopping with MediShop!',
    date: '3 days ago',
    read: true,
  },
];


const summaryCards = [
  { title: 'Total Orders', value: '12', icon: ShoppingCart },
  { title: 'Pending Delivery', value: '3', icon: Clock },
  { title: 'Cancelled Orders', value: '1', icon: XCircle },
  { title: 'Wishlist Items', value: '8', icon: Heart },
];

export default function DashboardPage() {
  const { user } = useUser();
  const [notifications, setNotifications] = React.useState(initialNotifications);

  const handleNotificationClick = (id: number) => {
    setNotifications(
      notifications.map((n) => (n.id === id ? { ...n, read: true } : n))
    );
  };

  return (
    <div className="p-4 md:p-8 space-y-6">
      <h1 className="text-3xl font-bold">
        Welcome back, {user?.displayName || 'User'}!
      </h1>

      {/* Summary Cards */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {summaryCards.map((item, index) => (
          <Card key={index}>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                {item.title}
              </CardTitle>
              <item.icon className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{item.value}</div>
            </CardContent>
          </Card>
        ))}
      </div>

      <div className="grid gap-6">
        {/* Notifications */}
        <div className="space-y-6">
           <h2 className="text-2xl font-bold">Notifications</h2>
          <Card>
            <CardHeader>
              <CardTitle className='flex items-center gap-2 text-xl'>
                <Bell className='h-5 w-5' />
                Activity Feed
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-1">
                {notifications.map((notification) => (
                  <Dialog key={notification.id} onOpenChange={(isOpen) => {
                    if (isOpen) handleNotificationClick(notification.id);
                  }}>
                    <DialogTrigger asChild>
                      <div className="flex items-start gap-4 p-3 rounded-lg hover:bg-muted/50 cursor-pointer transition-colors">
                        <div className="mt-1.5">
                          {!notification.read && (
                            <span className="flex h-2 w-2 rounded-full bg-primary" />
                          )}
                        </div>
                        <div className='flex-1'>
                          <p className={`text-sm ${!notification.read ? 'font-semibold' : ''}`}>
                            {notification.title}
                          </p>
                          <p className="text-xs text-muted-foreground mt-1">
                            {notification.date}
                          </p>
                        </div>
                         <Info className="h-4 w-4 text-muted-foreground mt-1" />
                      </div>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle>{notification.title}</DialogTitle>
                        <DialogDescription className="pt-2">
                          {notification.message}
                        </DialogDescription>
                      </DialogHeader>
                       {notification.cta && (
                        <div className="mt-4">
                          <Button asChild>
                             <Link href={notification.cta.href}>{notification.cta.text}</Link>
                          </Button>
                        </div>
                      )}
                    </DialogContent>
                  </Dialog>
                ))}
                {notifications.length === 0 && (
                  <p className="text-sm text-muted-foreground text-center py-4">No new notifications.</p>
                )}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}

