'use client';

import React, { useMemo } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import { Heart, AlertTriangle } from 'lucide-react';
import {
  useUser,
  useFirestore,
  useCollection,
  useMemoFirebase,
} from '@/firebase';
import { collection, query, where, doc, getDoc, getDocs } from 'firebase/firestore';
import type { WishlistItem, Product } from '@/lib/data';
import { ProductCard } from '@/components/product-card';
import { Skeleton } from '@/components/ui/skeleton';
import { PlaceHolderImages } from '@/lib/placeholder-images';

function WishlistGrid({ items }: { items: WishlistItem[] }) {
    const [products, setProducts] = React.useState<Product[]>([]);
    const [loading, setLoading] = React.useState(true);
    const firestore = useFirestore();

    React.useEffect(() => {
        const fetchProducts = async () => {
            if (items.length === 0) {
                setProducts([]);
                setLoading(false);
                return;
            }

            const productIds = items.map(item => item.productId);
            const productsRef = collection(firestore, 'products');
            // Firestore 'in' query is limited to 30 items. 
            // If the wishlist can be larger, this needs pagination or multiple queries.
            const productQuery = query(productsRef, where('id', 'in', productIds));

            try {
                const productSnapshots = await getDocs(productQuery);
                const fetchedProducts: Product[] = [];
                productSnapshots.forEach(doc => {
                    const data = doc.data() as Omit<Product, 'id'>;
                     // This is a workaround since product data is not fully stored in a 'products' collection.
                     // In a real app, you would fetch from a `products` collection.
                    const imageId = PlaceHolderImages.find(img => img.id.endsWith(doc.id))?.id || '';
                    fetchedProducts.push({ ...data, id: doc.id, imageId });
                });
                
                // A mock fetch since there's no "products" collection in the sample data
                const mockProducts: Product[] = [
                    { id: '1', name: 'Gentle Baby Lotion', price: 12.99, imageId: 'prod-1', category: 'Baby Care' },
                    { id: '2', name: "Men's Revitalizing Face Wash", price: 8.50, imageId: 'prod-2', category: "Men's Care" },
                    { id: '3', name: 'Vitamin C Plus Zinc', price: 19.99, imageId: 'prod-3', category: 'Supplements' },
                    { id: '4', name: 'Paracetamol 500mg Tablets', price: 2.99, imageId: 'prod-4', category: 'Medicines' },
                    { id: '5', name: 'Precision Digital Thermometer', price: 25.00, imageId: 'prod-5', category: 'Equipment' },
                    { id: '6', name: 'Ultra-thin Sanitary Pads', price: 5.49, imageId: 'prod-6', category: "Women's Care" },
                    { id: '7', name: 'Whey Protein Isolate', price: 45.99, imageId: 'prod-7', category: 'Supplements' },
                    { id: '8', name: 'Prefilled Insulin Syringe', price: 32.75, imageId: 'prod-8', category: 'Injections' },
                ];
                
                const wishlistedProducts = mockProducts.filter(p => productIds.includes(p.id));

                setProducts(wishlistedProducts);
            } catch (error) {
                console.error("Error fetching products:", error);
            } finally {
                setLoading(false);
            }
        };

        fetchProducts();
    }, [items, firestore]);

    if (loading) {
        return (
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                {[...Array(4)].map((_, i) => (
                    <Skeleton key={i} className="h-[350px] w-full" />
                ))}
            </div>
        );
    }
    
    if (products.length === 0) {
      return (
        <div className="flex flex-col items-center justify-center text-center text-muted-foreground bg-muted/40 p-10 rounded-lg">
          <Heart className="h-10 w-10 mb-4" />
          <p className="font-semibold">Your Wishlist is Empty</p>
          <p className="text-sm">
            Start adding products you love to see them here.
          </p>
        </div>
      )
    }

    return (
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {products.map(product => (
                <ProductCard key={product.id} product={product} />
            ))}
        </div>
    );
}

export default function WishlistPage() {
  const { user } = useUser();
  const firestore = useFirestore();

  const wishlistCollectionRef = useMemoFirebase(
    () => (user ? collection(firestore, 'users', user.uid, 'wishlist') : null),
    [firestore, user]
  );

  const {
    data: wishlistItems,
    isLoading,
    error,
  } = useCollection<WishlistItem>(wishlistCollectionRef);

  return (
    <div className="p-4 md:p-8">
      <h1 className="text-2xl font-bold mb-6">My Wishlist</h1>
      <Card>
        <CardContent className="pt-6">
          {isLoading && (
             <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                {[...Array(4)].map((_, i) => (
                    <Skeleton key={i} className="h-[350px] w-full" />
                ))}
            </div>
          )}

          {error && (
             <div className="flex flex-col items-center justify-center text-center text-destructive bg-destructive/10 p-6 rounded-lg">
                <AlertTriangle className="h-8 w-8 mb-2" />
                <p className="font-semibold">Failed to load wishlist.</p>
                <p className="text-sm">Please check your connection or try again later.</p>
            </div>
          )}

          {!isLoading && !error && wishlistItems && (
            <WishlistGrid items={wishlistItems} />
          )}

        </CardContent>
      </Card>
    </div>
  );
}