'use client';

import React, { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  RadioGroup,
  RadioGroupItem,
} from '@/components/ui/radio-group';
import { useToast } from '@/hooks/use-toast';
import { useAuth, useFirestore, useUser } from '@/firebase';
import { addAddress, updateAddress } from '@/firebase/firestore/addresses';
import type { Address } from '@/lib/data';
import { LoaderCircle } from 'lucide-react';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from './ui/form';

const addressSchema = z.object({
  fullName: z.string().min(3, 'Full name must be at least 3 characters.'),
  mobileNumber: z
    .string()
    .regex(/^[0-9]{10}$/, 'Please enter a valid 10-digit mobile number.'),
  houseNoStreet: z.string().min(5, 'Please enter a valid address.'),
  city: z.string().min(2, 'City is required.'),
  state: z.string().min(2, 'State is required.'),
  pincode: z.string().regex(/^[0-9]{6}$/, 'Please enter a valid 6-digit pincode.'),
  addressType: z.enum(['Home', 'Work'], {
    required_error: 'Please select an address type.',
  }),
});

type AddressFormValues = z.infer<typeof addressSchema>;

interface AddressFormProps {
  address?: Address;
  onSave?: () => void;
  onCancel?: () => void;
}

export function AddressForm({
  address,
  onSave,
  onCancel,
}: AddressFormProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();
  const { user } = useUser();
  const firestore = useFirestore();

  const form = useForm<AddressFormValues>({
    resolver: zodResolver(addressSchema),
    defaultValues: address
      ? {
          fullName: address.fullName,
          mobileNumber: address.mobileNumber,
          houseNoStreet: address.houseNoStreet,
          city: address.city,
          state: address.state,
          pincode: address.pincode,
          addressType: address.addressType,
        }
      : {
          fullName: '',
          mobileNumber: '',
          houseNoStreet: '',
          city: '',
          state: '',
          pincode: '',
          addressType: 'Home',
        },
  });

  const onSubmit = async (values: AddressFormValues) => {
    if (!user) {
      toast({
        variant: 'destructive',
        title: 'Not Authenticated',
        description: 'You must be logged in to save an address.',
      });
      return;
    }

    setIsSubmitting(true);
    const addressData: Omit<Address, 'id'> = {
      ...values,
      userId: user.uid,
    };

    try {
      if (address && address.id) {
        // Update existing address
        await updateAddress(firestore, user.uid, address.id, addressData);
        toast({
          title: 'Address Updated',
          description: 'Your address has been successfully updated.',
        });
      } else {
        // Add new address
        await addAddress(firestore, user.uid, addressData);
        toast({
          title: 'Address Saved',
          description: 'Your new address has been saved.',
        });
      }
      onSave?.();
    } catch (e: any) {
      toast({
        variant: 'destructive',
        title: 'Save Failed',
        description:
          e.message || 'There was an error saving your address. Please try again.',
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="fullName"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Full Name</FormLabel>
              <FormControl>
                <Input placeholder="John Doe" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="mobileNumber"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Mobile Number</FormLabel>
              <FormControl>
                <Input placeholder="123-456-7890" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="houseNoStreet"
          render={({ field }) => (
            <FormItem>
              <FormLabel>House No., Building, Street, Area</FormLabel>
              <FormControl>
                <Input placeholder="e.g. 21B, Baker Street" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <div className="grid grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="city"
            render={({ field }) => (
              <FormItem>
                <FormLabel>City</FormLabel>
                <FormControl>
                  <Input placeholder="e.g. Mumbai" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="state"
            render={({ field }) => (
              <FormItem>
                <FormLabel>State</FormLabel>
                <FormControl>
                  <Input placeholder="e.g. Maharashtra" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>
        <FormField
          control={form.control}
          name="pincode"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Pincode</FormLabel>
              <FormControl>
                <Input placeholder="e.g. 400001" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="addressType"
          render={({ field }) => (
            <FormItem className="space-y-3">
              <FormLabel>Address Type</FormLabel>
              <FormControl>
                <RadioGroup
                  onValueChange={field.onChange}
                  defaultValue={field.value}
                  className="flex items-center space-x-4"
                >
                  <FormItem className="flex items-center space-x-2 space-y-0">
                    <FormControl>
                      <RadioGroupItem value="Home" />
                    </FormControl>
                    <FormLabel className="font-normal">Home</FormLabel>
                  </FormItem>
                  <FormItem className="flex items-center space-x-2 space-y-0">
                    <FormControl>
                      <RadioGroupItem value="Work" />
                    </FormControl>
                    <FormLabel className="font-normal">Work</FormLabel>
                  </FormItem>
                </RadioGroup>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <div className="flex justify-end gap-2 pt-4">
          <Button type="button" variant="ghost" onClick={onCancel}>
            Cancel
          </Button>
          <Button type="submit" disabled={isSubmitting}>
            {isSubmitting && (
              <LoaderCircle className="animate-spin mr-2 h-4 w-4" />
            )}
            {address ? 'Save Changes' : 'Save Address'}
          </Button>
        </div>
      </form>
    </Form>
  );
}
