'use client';

import React, { useState } from 'react';
import Image from 'next/image';
import Link from 'next/link';
import { Button } from '@/components/ui/button';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import type { CartItem, Product } from '@/lib/data';
import { useToast } from '@/hooks/use-toast';
import { Trash2, LoaderCircle } from 'lucide-react';
import { useFirestore, useUser } from '@/firebase';
import { deleteDoc, doc, updateDoc } from 'firebase/firestore';

interface CartItemCardProps {
  item: CartItem;
  product: Product;
}

export function CartItemCard({ item, product }: CartItemCardProps) {
  const image = PlaceHolderImages.find((img) => img.id === product.imageId);
  const { toast } = useToast();
  const { user } = useUser();
  const firestore = useFirestore();
  const [isUpdating, setIsUpdating] = useState(false);
  const [isDeleting, setIsDeleting] = useState(false);

  const handleQuantityChange = async (newQuantity: number) => {
    if (!user || !item.id || newQuantity < 1) return;
    
    setIsUpdating(true);
    const cartItemRef = doc(firestore, 'users', user.uid, 'cart', item.id);
    
    try {
        await updateDoc(cartItemRef, { quantity: newQuantity });
    } catch(e: any) {
        toast({
            variant: 'destructive',
            title: 'Update failed',
            description: 'Could not update item quantity.'
        });
    } finally {
        setIsUpdating(false);
    }
  };

  const handleRemove = async () => {
    if (!user || !item.id) return;

    setIsDeleting(true);
    const cartItemRef = doc(firestore, 'users', user.uid, 'cart', item.id);
    
    try {
      await deleteDoc(cartItemRef);
      toast({
        title: 'Item Removed',
        description: `${product.name} has been removed from your cart.`,
      });
    } catch (e: any) {
      toast({
        variant: 'destructive',
        title: 'Removal failed',
        description: 'Could not remove item from cart.',
      });
    } finally {
      setIsDeleting(false);
    }
  };

  return (
    <div className="flex items-start gap-4 py-4">
      <div className="relative h-24 w-24 flex-shrink-0 overflow-hidden rounded-md">
        {image && (
          <Image
            src={image.imageUrl}
            alt={product.name}
            fill
            className="object-cover"
            data-ai-hint={image.imageHint}
          />
        )}
      </div>
      <div className="flex-grow">
        <Link href="#" className="font-semibold hover:underline">
          {product.name}
        </Link>
        <p className="text-sm text-muted-foreground">{product.category}</p>
        <p className="mt-1 text-lg font-bold">₹{product.price.toFixed(2)}</p>
        <div className="mt-2 flex items-center gap-2">
          <Button
            size="sm"
            variant="outline"
            onClick={() => handleQuantityChange(item.quantity - 1)}
            disabled={item.quantity <= 1 || isUpdating}
          >
            -
          </Button>
          <span>{item.quantity}</span>
          <Button
            size="sm"
            variant="outline"
            onClick={() => handleQuantityChange(item.quantity + 1)}
            disabled={isUpdating}
          >
            +
          </Button>
        </div>
      </div>
      <div className="flex flex-col items-end gap-2">
        <p className="text-lg font-bold">₹{(product.price * item.quantity).toFixed(2)}</p>
        <Button
          variant="ghost"
          size="icon"
          className="h-8 w-8 text-muted-foreground hover:text-destructive"
          onClick={handleRemove}
          disabled={isDeleting}
        >
          {isDeleting ? (
              <LoaderCircle className="h-4 w-4 animate-spin" />
          ) : (
              <Trash2 className="h-4 w-4" />
          )}
          <span className="sr-only">Remove item</span>
        </Button>
      </div>
    </div>
  );
}