'use client';

import React, { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useToast } from '@/hooks/use-toast';
import { useFirestore } from '@/firebase';
import { addCategory, updateCategory } from '@/firebase/firestore/categories';
import type { CategoryData } from '@/lib/data';
import { LoaderCircle } from 'lucide-react';
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from './ui/form';

const categorySchema = z.object({
  name: z.string().min(2, 'Category name must be at least 2 characters.'),
  icon: z.string().min(2, 'Icon name is required.'),
  bannerUrl: z.string().url('Please enter a valid URL.').optional().or(z.literal('')),
});

type CategoryFormValues = z.infer<typeof categorySchema>;

interface CategoryFormProps {
  category?: CategoryData;
  onSave?: () => void;
}

export function CategoryForm({ category, onSave }: CategoryFormProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();
  const firestore = useFirestore();

  const form = useForm<CategoryFormValues>({
    resolver: zodResolver(categorySchema),
    defaultValues: category
      ? {
          name: category.name,
          icon: category.icon,
          bannerUrl: category.bannerUrl || '',
        }
      : {
          name: '',
          icon: '',
          bannerUrl: '',
        },
  });

  const onSubmit = async (values: CategoryFormValues) => {
    setIsSubmitting(true);
    const categoryData = {
      ...values,
      bannerUrl: values.bannerUrl || '',
    };

    try {
      if (category && category.id) {
        await updateCategory(firestore, category.id, categoryData);
        toast({
          title: 'Category Updated',
          description: `The "${values.name}" category has been updated.`,
        });
      } else {
        await addCategory(firestore, categoryData);
        toast({
          title: 'Category Added',
          description: `The "${values.name}" category has been created.`,
        });
      }
      onSave?.();
    } catch (e: any) {
      // The error is already emitted to the global handler by the firestore function
      // We just need to inform the user that it failed.
      toast({
        variant: 'destructive',
        title: 'Save Failed',
        description: 'Could not save the category. Please try again.',
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="name"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Category Name</FormLabel>
              <FormControl>
                <Input placeholder="e.g., Baby Care" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="icon"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Icon Name</FormLabel>
              <FormControl>
                <Input placeholder="e.g., Baby" {...field} />
              </FormControl>
              <FormDescription>
                Enter a valid icon name from the lucide-react library.
              </FormDescription>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="bannerUrl"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Banner Image URL (Optional)</FormLabel>
              <FormControl>
                <Input type="url" placeholder="https://example.com/banner.jpg" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <div className="flex justify-end gap-2 pt-4">
          <Button type="button" variant="ghost" onClick={onSave}>
            Cancel
          </Button>
          <Button type="submit" disabled={isSubmitting}>
            {isSubmitting && <LoaderCircle className="animate-spin mr-2 h-4 w-4" />}
            {category ? 'Save Changes' : 'Create Category'}
          </Button>
        </div>
      </form>
    </Form>
  );
}

    