'use client';

import Link from 'next/link';
import { CategoryData } from '@/lib/data';
import { motion } from 'framer-motion';
import * as LucideIcons from 'lucide-react';
import { Skeleton } from './ui/skeleton';
import { createElement } from 'react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection } from 'firebase/firestore';

// A type guard to check if a string is a valid Lucide icon name
function isLucideIcon(name: string): name is keyof typeof LucideIcons {
  return name in LucideIcons;
}

function CategoryIcon({ name }: { name: string }) {
  if (isLucideIcon(name)) {
    return createElement(LucideIcons[name], {
      className: 'h-12 w-12 text-primary',
    });
  }
  // Fallback icon if the name is invalid
  return <LucideIcons.Package className="h-12 w-12 text-primary" />;
}

export function CategorySlider() {
  const firestore = useFirestore();
  const categoriesCollectionRef = useMemoFirebase(
    () => collection(firestore, 'categories'),
    [firestore]
  );
  const { data: categories, isLoading } =
    useCollection<CategoryData>(categoriesCollectionRef);

  if (isLoading || !categories) {
    return (
      <section id="categories" className="bg-background py-12 md:py-16">
        <div className="container flex flex-col items-center">
          <h2 className="text-2xl md:text-3xl font-bold tracking-tight text-center font-headline mb-8 md:mb-12">
            Shop by Category
          </h2>
          <div className="grid grid-cols-2 sm:grid-cols-4 lg:grid-cols-8 gap-4 md:gap-6 justify-center">
            {[...Array(8)].map((_, i) => (
              <div key={i} className="flex flex-col items-center gap-3">
                <Skeleton className="h-28 w-28 rounded-full" />
                <Skeleton className="h-4 w-20" />
              </div>
            ))}
          </div>
        </div>
      </section>
    );
  }

  return (
    <section id="categories" className="bg-background py-12 md:py-16">
      <div className="container flex flex-col items-center">
        <h2 className="text-2xl md:text-3xl font-bold tracking-tight text-center font-headline mb-8 md:mb-12">
          Shop by Category
        </h2>
        <div className="grid grid-cols-2 sm:grid-cols-4 lg:grid-cols-8 gap-4 md:gap-6 justify-center">
          {categories?.map((category, index) => (
            <Link href="#" key={category.id} className="group">
              <motion.div
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.3, delay: index * 0.05 }}
                className="flex flex-col items-center gap-3 text-center"
              >
                <div className="flex h-28 w-28 items-center justify-center rounded-full bg-muted/50 group-hover:bg-primary/10 transition-all duration-300 border-2 border-transparent group-hover:border-primary/50">
                  <CategoryIcon name={category.icon} />
                </div>
                <span className="text-sm font-semibold text-foreground group-hover:text-primary">
                  {category.name}
                </span>
              </motion.div>
            </Link>
          ))}
        </div>
      </div>
    </section>
  );
}
