'use client';

import React, { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useToast } from '@/hooks/use-toast';
import { useFirestore } from '@/firebase';
import { addDoc, collection, doc, serverTimestamp, setDoc, updateDoc } from 'firebase/firestore';
import type { Coupon } from '@/lib/data';
import { Calendar as CalendarIcon, LoaderCircle } from 'lucide-react';
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from './ui/form';
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from '@/components/ui/popover';
import { Calendar } from './ui/calendar';
import { cn } from '@/lib/utils';
import { format } from 'date-fns';

const couponSchema = z.object({
  code: z.string().min(4, 'Code must be at least 4 characters.').toUpperCase(),
  discountPercentage: z.coerce.number().min(1, 'Discount must be at least 1%').max(100, 'Discount cannot exceed 100%'),
  minOrderAmount: z.coerce.number().nonnegative('Minimum order amount cannot be negative.'),
  maxUsageLimit: z.coerce.number().int().positive('Usage limit must be at least 1.'),
  expiryDate: z.date({
    required_error: 'An expiry date is required.',
  }),
});

type CouponFormValues = z.infer<typeof couponSchema>;

interface CouponFormProps {
  coupon?: Coupon;
  onSave?: () => void;
}

export function CouponForm({ coupon, onSave }: CouponFormProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();
  const firestore = useFirestore();

  const form = useForm<CouponFormValues>({
    resolver: zodResolver(couponSchema),
    defaultValues: coupon
      ? {
          ...coupon,
          expiryDate: coupon.expiryDate?.seconds ? new Date(coupon.expiryDate.seconds * 1000) : new Date(),
        }
      : {
          code: '',
          discountPercentage: 10,
          minOrderAmount: 0,
          maxUsageLimit: 100,
          expiryDate: new Date(),
        },
  });

  const onSubmit = async (values: CouponFormValues) => {
    setIsSubmitting(true);
    
    try {
        const couponData = {
            ...values,
            createdAt: serverTimestamp(),
        };

      if (coupon && coupon.id) {
        const couponRef = doc(firestore, 'coupons', coupon.id);
        await updateDoc(couponRef, couponData);
        toast({
          title: 'Coupon Updated',
          description: `The "${values.code}" coupon has been updated.`,
        });
      } else {
        const couponRef = collection(firestore, 'coupons');
        await addDoc(couponRef, couponData);
        toast({
          title: 'Coupon Created',
          description: `The "${values.code}" coupon has been created.`,
        });
      }
      onSave?.();
    } catch (e: any) {
      toast({
        variant: 'destructive',
        title: 'Save Failed',
        description: 'Could not save the coupon. Please try again.',
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="code"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Coupon Code</FormLabel>
              <FormControl>
                <Input placeholder="e.g., SUMMER25" {...field} />
              </FormControl>
              <FormDescription>This is the code customers will enter at checkout.</FormDescription>
              <FormMessage />
            </FormItem>
          )}
        />

        <div className="grid grid-cols-2 gap-4">
            <FormField
            control={form.control}
            name="discountPercentage"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Discount (%)</FormLabel>
                <FormControl>
                    <Input type="number" placeholder="e.g., 25" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
            />
            <FormField
            control={form.control}
            name="minOrderAmount"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Min. Order Amount (₹)</FormLabel>
                <FormControl>
                    <Input type="number" placeholder="e.g., 500" {...field} />
                </FormControl>
                <FormMessage />
                </FormItem>
            )}
            />
        </div>

        <div className="grid grid-cols-2 gap-4">
            <FormField
                control={form.control}
                name="maxUsageLimit"
                render={({ field }) => (
                    <FormItem>
                    <FormLabel>Max Usage Limit</FormLabel>
                    <FormControl>
                        <Input type="number" placeholder="e.g., 1000" {...field} />
                    </FormControl>
                    <FormMessage />
                    </FormItem>
                )}
            />
            <FormField
                control={form.control}
                name="expiryDate"
                render={({ field }) => (
                <FormItem className="flex flex-col">
                    <FormLabel>Expiry Date</FormLabel>
                    <Popover>
                    <PopoverTrigger asChild>
                        <FormControl>
                        <Button
                            variant={'outline'}
                            className={cn(
                            'w-full pl-3 text-left font-normal',
                            !field.value && 'text-muted-foreground'
                            )}
                        >
                            {field.value ? (
                            format(field.value, 'PPP')
                            ) : (
                            <span>Pick a date</span>
                            )}
                            <CalendarIcon className="ml-auto h-4 w-4 opacity-50" />
                        </Button>
                        </FormControl>
                    </PopoverTrigger>
                    <PopoverContent className="w-auto p-0" align="start">
                        <Calendar
                        mode="single"
                        selected={field.value}
                        onSelect={field.onChange}
                        disabled={(date) => date < new Date()}
                        initialFocus
                        />
                    </PopoverContent>
                    </Popover>
                    <FormMessage />
                </FormItem>
                )}
            />
        </div>
        
        <div className="flex justify-end gap-2 pt-4">
            <Button type="button" variant="ghost" onClick={onSave}>
                Cancel
            </Button>
            <Button type="submit" disabled={isSubmitting}>
                {isSubmitting && <LoaderCircle className="animate-spin mr-2 h-4 w-4" />}
                {coupon ? 'Save Changes' : 'Create Coupon'}
            </Button>
        </div>
      </form>
    </Form>
  );
}
