
'use client';

import React, { useState, useRef } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useToast } from '@/hooks/use-toast';
import { useFirestore } from '@/firebase';
import type { HeroContent } from '@/lib/data';
import { LoaderCircle, Upload } from 'lucide-react';
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from './ui/form';
import { Switch } from './ui/switch';
import {
  addDoc,
  collection,
  doc,
  serverTimestamp,
  updateDoc,
} from 'firebase/firestore';

const heroContentSchema = z.object({
  headline: z.string().min(10, 'Headline must be at least 10 characters.'),
  subheading: z.string().min(10, 'Subheading must be at least 10 characters.'),
  cta1: z.string().min(3, 'CTA text is required.'),
  cta2: z.string().min(3, 'CTA text is required.'),
  imageId: z.string().min(1, 'Image ID is required (e.g., hero-1).'),
  imageUrl: z.string().url('Please enter a valid URL.').optional().or(z.literal('')),
  order: z.coerce.number().int().min(0),
  isActive: z.boolean().default(true),
});

type HeroContentFormValues = z.infer<typeof heroContentSchema>;

interface HeroContentFormProps {
  slide?: HeroContent;
  onSave?: () => void;
}

export function HeroContentForm({ slide, onSave }: HeroContentFormProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();
  const firestore = useFirestore();
  const fileInputRef = useRef<HTMLInputElement>(null);

  const form = useForm<HeroContentFormValues>({
    resolver: zodResolver(heroContentSchema),
    defaultValues: slide
      ? { ...slide, imageUrl: slide.imageUrl || '' }
      : {
          headline: '',
          subheading: '',
          cta1: 'Shop Now',
          cta2: 'Learn More',
          imageId: 'hero-1',
          imageUrl: '',
          order: 0,
          isActive: true,
        },
  });

  const handleUploadClick = () => {
    fileInputRef.current?.click();
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // In a real app, you'd upload this file to Firebase Storage and get a URL.
      // For now, we'll just show a toast.
      toast({
        title: 'Image Selected (Demo)',
        description: `"${file.name}" is ready. Save to link it (this is a demo and will not actually upload).`,
      });
      // Optionally, you could use FileReader to display a preview.
      // And then set the URL field, but that requires more state management.
    }
  };


  const onSubmit = async (values: HeroContentFormValues) => {
    setIsSubmitting(true);
    const contentData = {
      ...values,
      updatedAt: serverTimestamp(),
    };

    try {
      if (slide && slide.id) {
        const slideRef = doc(firestore, 'hero_content', slide.id);
        await updateDoc(slideRef, contentData);
        toast({
          title: 'Slide Updated',
          description: 'The hero slide has been successfully updated.',
        });
      } else {
        const collectionRef = collection(firestore, 'hero_content');
        await addDoc(collectionRef, { ...contentData, createdAt: serverTimestamp() });
        toast({
          title: 'Slide Created',
          description: 'The new hero slide has been added.',
        });
      }
      onSave?.();
    } catch (e: any) {
      toast({
        variant: 'destructive',
        title: 'Save Failed',
        description: 'Could not save the slide. Please try again.',
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="headline"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Headline</FormLabel>
              <FormControl>
                <Input placeholder="e.g., Reliable Pharmacy Solutions..." {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="subheading"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Subheading</FormLabel>
              <FormControl>
                <Input placeholder="e.g., Get genuine medicines..." {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <div className="grid grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="cta1"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Primary Button Text</FormLabel>
                <FormControl>
                  <Input {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          <FormField
            control={form.control}
            name="cta2"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Secondary Button Text</FormLabel>
                <FormControl>
                  <Input {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>
        
        <FormField
            control={form.control}
            name="imageUrl"
            render={({ field }) => (
                <FormItem>
                <FormLabel>Banner Image URL</FormLabel>
                <FormControl>
                    <div className="flex items-center gap-2">
                        <Input type="url" placeholder="https://example.com/banner.jpg" {...field} />
                         <input
                            type="file"
                            ref={fileInputRef}
                            onChange={handleFileChange}
                            className="hidden"
                            accept="image/*"
                         />
                        <Button type="button" variant="outline" size="icon" onClick={handleUploadClick}>
                            <Upload className="h-4 w-4" />
                            <span className="sr-only">Upload</span>
                        </Button>
                    </div>
                </FormControl>
                <FormDescription>Enter a direct link to a banner image, or upload one.</FormDescription>
                <FormMessage />
                </FormItem>
            )}
        />

        <div className="grid grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="imageId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Fallback Image ID</FormLabel>
                <FormControl>
                  <Input placeholder="hero-1, hero-2, etc." {...field} />
                </FormControl>
                <FormDescription>From placeholder-images.json (used if URL is empty).</FormDescription>
                <FormMessage />
              </FormItem>
            )}
          />
           <FormField
            control={form.control}
            name="order"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Display Order</FormLabel>
                <FormControl>
                  <Input type="number" {...field} />
                </FormControl>
                 <FormDescription>Lower numbers appear first.</FormDescription>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

         <FormField
            control={form.control}
            name="isActive"
            render={({ field }) => (
                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3 shadow-sm">
                <div className="space-y-0.5">
                    <FormLabel>Is Active</FormLabel>
                    <FormDescription>
                        Determines if this slide is visible on the homepage.
                    </FormDescription>
                </div>
                <FormControl>
                    <Switch
                    checked={field.value}
                    onCheckedChange={field.onChange}
                    />
                </FormControl>
                </FormItem>
            )}
        />


        <div className="flex justify-end gap-2 pt-4">
          <Button type="button" variant="ghost" onClick={onSave}>
            Cancel
          </Button>
          <Button type="submit" disabled={isSubmitting}>
            {isSubmitting && <LoaderCircle className="animate-spin mr-2 h-4 w-4" />}
            {slide ? 'Save Changes' : 'Create Slide'}
          </Button>
        </div>
      </form>
    </Form>
  );
}
