'use client';

import Image from 'next/image';
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
} from '@/components/ui/carousel';
import { Button } from '@/components/ui/button';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import Autoplay from 'embla-carousel-autoplay';
import { motion } from 'framer-motion';
import Link from 'next/link';
import type { HeroContent } from '@/lib/data';
import { Skeleton } from './ui/skeleton';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, query, where } from 'firebase/firestore';

export function HeroSection() {
  const firestore = useFirestore();
  const heroContentCollectionRef = useMemoFirebase(
    () =>
      query(
        collection(firestore, 'hero_content'),
        where('isActive', '==', true)
      ),
    [firestore]
  );
  const { data: slides, isLoading } =
    useCollection<HeroContent>(heroContentCollectionRef);

  if (isLoading) {
    return (
      <section className="w-full border-b">
        <Skeleton className="h-[60vh] md:h-[calc(100vh-4rem)] w-full" />
      </section>
    );
  }

  // If there are no slides, don't render the section
  if (!slides || slides.length === 0) {
    return null;
  }

  return (
    <section className="w-full border-b">
      <Carousel
        opts={{ loop: true }}
        plugins={[Autoplay({ delay: 5000, stopOnInteraction: true })]}
        className="relative group"
      >
        <CarouselContent>
          {slides.map((content, index) => {
            const image = content.imageUrl
              ? {
                  imageUrl: content.imageUrl,
                  description: content.headline,
                  imageHint: 'custom banner',
                }
              : PlaceHolderImages.find((img) => img.id === content.imageId);

            return (
              <CarouselItem key={content.id || index}>
                <div className="relative h-[60vh] md:h-[calc(100vh-4rem)] w-full">
                  {image && (
                    <Image
                      src={image.imageUrl}
                      alt={image.description}
                      fill
                      className="object-cover"
                      data-ai-hint={image.imageHint}
                      priority={index === 0}
                    />
                  )}
                  <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-black/30 to-transparent" />
                  <div className="absolute inset-0 flex items-center">
                    <motion.div
                      key={content.id} // Add key for re-animation on slide change
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.5, delay: 0.2 }}
                      className="container text-center md:text-left"
                    >
                      <div className="bg-black/20 md:bg-transparent p-6 rounded-lg md:p-0">
                        <h1 className="text-3xl md:text-5xl font-bold tracking-tight font-headline max-w-2xl text-white">
                          {content.headline}
                        </h1>
                        <p className="mt-4 max-w-lg text-base md:text-lg text-white/90">
                          {content.subheading}
                        </p>
                        <div className="mt-8 flex flex-col sm:flex-row justify-center md:justify-start gap-4">
                          <Button size="lg" asChild>
                            <Link href="#">{content.cta1}</Link>
                          </Button>
                          <Button size="lg" variant="secondary" asChild>
                            <Link href="#">{content.cta2}</Link>
                          </Button>
                        </div>
                      </div>
                    </motion.div>
                  </div>
                </div>
              </CarouselItem>
            );
          })}
        </CarouselContent>
        <CarouselPrevious className="absolute left-4 top-1/2 -translate-y-1/2 text-white bg-white/20 hover:bg-white/30 border-none opacity-0 group-hover:opacity-100 transition-opacity" />
        <CarouselNext className="absolute right-4 top-1/2 -translate-y-1/2 text-white bg-white/20 hover:bg-white/30 border-none opacity-0 group-hover:opacity-100 transition-opacity" />
      </Carousel>
    </section>
  );
}
