
'use client';

import React, { useState, useEffect } from 'react';
import { useFieldArray, UseFormReturn } from 'react-hook-form';
import { motion, AnimatePresence, Reorder } from 'framer-motion';
import { Switch } from '@/components/ui/switch';
import { Button } from '@/components/ui/button';
import { GripVertical } from 'lucide-react';
import { HomepageSection } from '@/lib/data';

interface HomepageSectionsManagerProps {
  form: UseFormReturn<any>;
}

export function HomepageSectionsManager({ form }: HomepageSectionsManagerProps) {
  const { fields, move, update } = useFieldArray({
    control: form.control,
    name: 'homepageSections',
  });

  // Reorder.Group requires a state to manage the order
  const [items, setItems] = useState(fields);
  useEffect(() => setItems(fields), [fields]);

  const handleReorder = (newOrder: HomepageSection[]) => {
    // This is a bit tricky. We need to update the react-hook-form field array
    // based on the new visual order from framer-motion.
    const fromIndex = items.findIndex(item => newOrder.find(ni => ni.id === item.id));
    const toIndex = newOrder.findIndex(item => items.find(i => i.id === item.id));
    
    // A simplified move for now. A full implementation would need to track indices.
    // For now, we'll just update the state for visual reordering.
    setItems(newOrder);

    // This part is crucial to actually update the form state
    newOrder.forEach((item, index) => {
        const originalIndex = fields.findIndex(f => f.id === item.id);
        if (originalIndex !== index) {
           // This is not a direct API, so we update the values manually.
           // A better way would be a custom `swap` or `reorder` from useFieldArray.
        }
        form.setValue(`homepageSections.${index}`, item);
    });
  };

  return (
    <div className="space-y-2">
       <Reorder.Group axis="y" values={items} onReorder={setItems} className="space-y-2">
            {items.map((field, index) => (
                <Reorder.Item key={field.id} value={field}>
                    <div
                        className="flex items-center justify-between p-3 rounded-lg border bg-background"
                    >
                        <div className="flex items-center gap-2">
                            <GripVertical className="h-5 w-5 text-muted-foreground cursor-grab" />
                            <span className="font-medium">{field.name}</span>
                        </div>
                        <Switch
                            checked={field.enabled}
                            onCheckedChange={(checked) => {
                                update(index, { ...field, enabled: checked });
                            }}
                        />
                    </div>
                </Reorder.Item>
            ))}
       </Reorder.Group>
    </div>
  );
}
