'use client';

import { useState, useTransition } from 'react';
import { handleIntelligentSearch } from '@/lib/actions';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { LoaderCircle, Search } from 'lucide-react';
import { Card, CardContent } from '@/components/ui/card';
import { useToast } from '@/hooks/use-toast';

export function IntelligentSearch() {
  const [query, setQuery] = useState('');
  const [results, setResults] = useState<string[]>([]);
  const [isPending, startTransition] = useTransition();
  const { toast } = useToast();

  const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    if (!query.trim()) return;

    startTransition(async () => {
      const result = await handleIntelligentSearch(query);
      if (result.error) {
        toast({
          variant: 'destructive',
          title: 'Search Error',
          description: result.error,
        });
        setResults([]);
      } else {
        setResults(result.suggestions || []);
      }
    });
  };

  return (
    <div className="relative w-full">
      <form onSubmit={handleSubmit} className="flex w-full items-center">
        <Input
          type="search"
          placeholder="Search for medicines and products..."
          className="pr-12"
          value={query}
          onChange={(e) => setQuery(e.target.value)}
        />
        <Button
          type="submit"
          size="icon"
          variant="ghost"
          className="absolute right-1 top-1/2 -translate-y-1/2 h-8 w-8"
          disabled={isPending}
        >
          {isPending ? (
            <LoaderCircle className="h-4 w-4 animate-spin" />
          ) : (
            <Search className="h-4 w-4" />
          )}
          <span className="sr-only">Search</span>
        </Button>
      </form>
      {isPending && (
        <Card className="absolute top-full mt-2 w-full z-10">
          <CardContent className="p-4">
            <div className="flex items-center justify-center space-x-2 text-sm text-muted-foreground">
              <LoaderCircle className="h-4 w-4 animate-spin" />
              <span>Searching...</span>
            </div>
          </CardContent>
        </Card>
      )}
      {!isPending && results.length > 0 && (
        <Card className="absolute top-full mt-2 w-full z-10">
          <CardContent className="p-2">
            <ul className="space-y-1">
              {results.map((item, index) => (
                <li key={index} className="p-2 text-sm rounded-md hover:bg-accent/50">
                  {item}
                </li>
              ))}
            </ul>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
