'use client';

import { useState, useTransition, useRef, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { LoaderCircle, Bot, Send, User, X } from 'lucide-react';
import { handleLiveChat } from '@/lib/actions';
import { cn } from '@/lib/utils';
import { Avatar, AvatarFallback } from './ui/avatar';
import { ScrollArea } from './ui/scroll-area';

const chatCategories = [
  'Order Issues',
  'Payment & Refunds',
  'Delivery Problems',
  'Product Information',
  'Account Issues',
];

interface Message {
  sender: 'user' | 'bot';
  text: string;
}

export function LiveChat() {
  const [category, setCategory] = useState<string>('');
  const [chatStarted, setChatStarted] = useState(false);
  const [messages, setMessages] = useState<Message[]>([]);
  const [input, setInput] = useState('');
  const [isPending, startTransition] = useTransition();
  const scrollViewportRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (scrollViewportRef.current) {
      scrollViewportRef.current.scrollTo({
        top: scrollViewportRef.current.scrollHeight,
        behavior: 'smooth'
      });
    }
  }, [messages, isPending]);

  const handleStartChat = () => {
    if (category) {
      setChatStarted(true);
      setMessages([
        { sender: 'bot', text: `Hi there! How can I help you with ${category} today?` },
      ]);
    }
  };

  const handleCloseChat = () => {
    setChatStarted(false);
    setMessages([]);
    setCategory('');
    setInput('');
  }

  const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    if (!input.trim()) return;

    const userMessage: Message = { sender: 'user', text: input };
    setMessages((prev) => [...prev, userMessage]);
    setInput('');

    startTransition(async () => {
      const botResponse = await handleLiveChat(input, category);
      const botMessage: Message = { sender: 'bot', text: botResponse };
      setMessages((prev) => [...prev, botMessage]);
    });
  };

  return (
    <Card className="w-full">
      <CardHeader className="flex flex-row items-start justify-between gap-3">
        <div className='flex items-center gap-3'>
          <Bot className="h-8 w-8 text-primary" />
          <div>
            <CardTitle>24/7 Virtual Support Assistant</CardTitle>
            <CardDescription>Get instant help with your questions.</CardDescription>
          </div>
        </div>
        {chatStarted && (
            <Button variant="ghost" size="icon" onClick={handleCloseChat} className="h-8 w-8 flex-shrink-0">
                <X className="h-4 w-4" />
                <span className="sr-only">Close chat</span>
            </Button>
        )}
      </CardHeader>
      <CardContent>
        {!chatStarted ? (
          <div className="space-y-4">
            <p className="text-sm text-muted-foreground">
              Please select a category to start the chat.
            </p>
            <Select onValueChange={setCategory} value={category}>
              <SelectTrigger>
                <SelectValue placeholder="Select a category..." />
              </SelectTrigger>
              <SelectContent>
                {chatCategories.map((cat) => (
                  <SelectItem key={cat} value={cat}>
                    {cat}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            <Button onClick={handleStartChat} disabled={!category} className="w-full">
              Start Chat
            </Button>
          </div>
        ) : (
          <div className="flex flex-col h-96">
            <ScrollArea className="flex-1" viewportRef={scrollViewportRef}>
                <div className="space-y-4 p-4 pr-6">
              {messages.map((message, index) => (
                <div
                  key={index}
                  className={cn(
                    'flex items-start gap-3 my-4',
                    message.sender === 'user' ? 'justify-end' : ''
                  )}
                >
                  {message.sender === 'bot' && (
                    <Avatar className="h-8 w-8 bg-primary text-primary-foreground">
                       <AvatarFallback><Bot size={18} /></AvatarFallback>
                    </Avatar>
                  )}
                  <div
                    className={cn(
                      'rounded-lg p-3 max-w-sm',
                      message.sender === 'user'
                        ? 'bg-primary text-primary-foreground'
                        : 'bg-muted'
                    )}
                  >
                    <p className="text-sm">{message.text}</p>
                  </div>
                   {message.sender === 'user' && (
                     <Avatar className="h-8 w-8">
                       <AvatarFallback><User size={18} /></AvatarFallback>
                     </Avatar>
                  )}
                </div>
              ))}
               {isPending && (
                <div className="flex items-start gap-3 my-4">
                   <Avatar className="h-8 w-8 bg-primary text-primary-foreground">
                        <AvatarFallback><Bot size={18} /></AvatarFallback>
                    </Avatar>
                    <div className="rounded-lg p-3 max-w-sm bg-muted flex items-center">
                        <LoaderCircle className="h-4 w-4 animate-spin text-muted-foreground" />
                    </div>
                </div>
              )}
              </div>
            </ScrollArea>
            <form onSubmit={handleSubmit} className="flex items-center gap-2 border-t p-4">
              <Input
                value={input}
                onChange={(e) => setInput(e.target.value)}
                placeholder="Type your message..."
                disabled={isPending}
              />
              <Button type="submit" size="icon" disabled={isPending || !input.trim()}>
                <Send className="h-4 w-4" />
                <span className="sr-only">Send</span>
              </Button>
            </form>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
