'use client';

import Image from 'next/image';
import Link from 'next/link';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardFooter, CardHeader } from '@/components/ui/card';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import type { Order } from '@/lib/data';
import { OrderTimeline } from './order-timeline';
import { useToast } from '@/hooks/use-toast';

interface OrderCardProps {
  order: Order;
}

export function OrderCard({ order }: OrderCardProps) {
  const image = PlaceHolderImages.find((img) => img.id === order.product.imageId);
  const { toast } = useToast();

  const handleDownloadInvoice = () => {
    toast({
      title: 'Invoice Download',
      description: 'Your invoice is being prepared for download.',
    });
  };
  return (
    <Card className="overflow-hidden">
      <CardHeader className="p-4 bg-muted/40 border-b">
        <div className="flex flex-col sm:flex-row justify-between sm:items-center gap-2">
          <div className="text-sm text-muted-foreground">
            <span className="font-semibold text-foreground">Order ID:</span> #{order.id}
          </div>
          <div className="text-sm text-muted-foreground">
            <span className="font-semibold text-foreground">Ordered on:</span>{' '}
            {new Date(order.orderDate).toLocaleDateString('en-US', {
              year: 'numeric',
              month: 'long',
              day: 'numeric',
            })}
          </div>
        </div>
      </CardHeader>
      <CardContent className="p-4">
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="flex-shrink-0">
            <div className="relative h-24 w-24 rounded-md overflow-hidden">
              {image && (
                <Image
                  src={image.imageUrl}
                  alt={order.product.name}
                  fill
                  className="object-cover"
                  data-ai-hint={image.imageHint}
                />
              )}
            </div>
          </div>
          <div className="flex-grow">
            <h3 className="font-semibold text-lg">{order.product.name}</h3>
            <p className="text-muted-foreground text-sm">Category: {order.product.category}</p>
            <p className="font-bold text-primary mt-1">₹{order.product.price.toFixed(2)}</p>
          </div>
          <div className="flex-shrink-0 flex sm:flex-col justify-between items-end sm:items-center pt-2">
             <div className="text-center">
                 <p className="font-semibold text-sm">
                    {order.status === 'Delivered' ? 'Delivered on' : 'Expected Delivery'}
                 </p>
                 <p className="text-sm text-muted-foreground">
                    {new Date(order.expectedDelivery).toLocaleDateString('en-US', { month: 'short', day: 'numeric' })}
                 </p>
            </div>
          </div>
        </div>
        <OrderTimeline
          currentStatus={order.status}
          courier={order.courier}
          trackingId={order.trackingId}
        />
      </CardContent>
      <CardFooter className="p-4 bg-muted/40 border-t flex justify-end gap-2">
        <Button variant="outline" asChild>
          <Link href={`/dashboard/orders/${order.id}`}>View Order Details</Link>
        </Button>
        <Button onClick={handleDownloadInvoice}>Download Invoice</Button>
      </CardFooter>
    </Card>
  );
}
