'use client';

import { cn } from '@/lib/utils';
import type { OrderStatus } from '@/lib/data';
import { Check, Package, Rocket, Truck, Home, Copy } from 'lucide-react';
import { motion, AnimatePresence } from 'framer-motion';
import { useToast } from '@/hooks/use-toast';
import { Button } from './ui/button';

const statuses: {
  name: OrderStatus;
  icon: React.ComponentType<any>;
}[] = [
  { name: 'Order Placed', icon: Check },
  { name: 'Packed', icon: Package },
  { name: 'Shipped', icon: Rocket },
  { name: 'Out for Delivery', icon: Truck },
  { name: 'Delivered', icon: Home },
];

interface OrderTimelineProps {
  currentStatus: OrderStatus;
  courier?: string;
  trackingId?: string;
}

export function OrderTimeline({ currentStatus, courier, trackingId }: OrderTimelineProps) {
  const currentIndex = statuses.findIndex((s) => s.name === currentStatus);
  const shippedIndex = statuses.findIndex((s) => s.name === 'Shipped');
  const isShippedOrBeyond = currentIndex >= shippedIndex;

  const { toast } = useToast();

  const handleCopy = (text: string) => {
    navigator.clipboard.writeText(text);
    toast({
      title: 'Copied to Clipboard',
      description: text,
    });
  };

  return (
    <div className="mt-6">
      <div className="relative flex justify-between items-start w-full">
        <div className="absolute left-0 top-5 h-1 w-full -translate-y-1/2 bg-muted">
          <motion.div
            className="h-full bg-primary"
            initial={{ width: 0 }}
            animate={{ width: `${(currentIndex / (statuses.length - 1)) * 100}%` }}
            transition={{ duration: 0.5, ease: 'easeInOut' }}
          />
        </div>

        {statuses.map((status, index) => {
          const isCompleted = index <= currentIndex;
          return (
            <div key={status.name} className="relative z-10 flex flex-col items-center group">
              <motion.div
                className={cn(
                  'flex h-10 w-10 items-center justify-center rounded-full border-2 transition-all duration-300',
                  isCompleted
                    ? 'bg-primary border-primary text-primary-foreground'
                    : 'bg-background border-muted-foreground/50 text-muted-foreground'
                )}
                whileHover={{ scale: 1.1 }}
              >
                <status.icon className="h-5 w-5" />
              </motion.div>
              <p
                className={cn(
                  'text-xs text-center mt-2 w-20',
                  isCompleted ? 'font-semibold text-foreground' : 'text-muted-foreground'
                )}
              >
                {status.name}
              </p>
            </div>
          );
        })}
      </div>
      <AnimatePresence>
        {isShippedOrBeyond && courier && trackingId && (
          <motion.div
            initial={{ opacity: 0, y: 10 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: 10 }}
            transition={{ duration: 0.3, delay: 0.3 }}
            className="mt-4 flex justify-center items-center gap-4 text-sm bg-muted/50 p-3 rounded-lg"
          >
            <div className="flex items-center gap-2">
              <span className="font-semibold">Courier:</span>
              <span className="text-muted-foreground">{courier}</span>
            </div>
            <div className="h-4 w-px bg-border" />
            <div className="flex items-center gap-2">
              <span className="font-semibold">Tracking ID:</span>
              <span className="text-muted-foreground font-mono">{trackingId}</span>
              <Button size="icon" variant="ghost" className="h-6 w-6" onClick={() => handleCopy(trackingId)}>
                <Copy className="h-3 w-3" />
                <span className="sr-only">Copy Tracking ID</span>
              </Button>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  );
}
