'use client';

import Image from 'next/image';
import { motion } from 'framer-motion';
import { Eye, Heart, ShoppingCart, Star } from 'lucide-react';

import { Card, CardContent, CardFooter, CardHeader } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import type { Product } from '@/lib/data';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import { useToast } from '@/hooks/use-toast';
import { useUser, useFirestore } from '@/firebase';
import { addDoc, collection, serverTimestamp, query, where, getDocs, deleteDoc, doc, runTransaction } from 'firebase/firestore';
import { Badge } from './ui/badge';

interface ProductCardProps {
  product: Product;
}

export function ProductCard({ product }: ProductCardProps) {
  const { toast } = useToast();
  const image = PlaceHolderImages.find((img) => img.id === product.imageId);
  const { user } = useUser();
  const firestore = useFirestore();

  const handleAddToCart = async () => {
     if (!user) {
      toast({
        variant: 'destructive',
        title: 'Not Logged In',
        description: 'You must be logged in to add items to your cart.',
      });
      return;
    }

    const cartRef = collection(firestore, 'users', user.uid, 'cart');
    
    try {
        await runTransaction(firestore, async (transaction) => {
            const q = query(cartRef, where("productId", "==", product.id));
            const querySnapshot = await getDocs(q);

            if (!querySnapshot.empty) {
                // Product is already in the cart, so update quantity
                const docToUpdate = querySnapshot.docs[0];
                const newQuantity = docToUpdate.data().quantity + 1;
                transaction.update(docToUpdate.ref, { quantity: newQuantity });
                 toast({
                    title: 'Quantity Updated',
                    description: `${product.name} quantity increased in your cart.`,
                });
            } else {
                 // Product is not in the cart, so add it
                const cartItem = {
                    userId: user.uid,
                    productId: product.id,
                    quantity: 1,
                    addedAt: serverTimestamp(),
                };
                transaction.set(doc(cartRef), cartItem);
                 toast({
                    title: 'Added to Cart',
                    description: `${product.name} has been added to your cart.`,
                });
            }
        });
    } catch (e) {
        console.error("Transaction failed: ", e);
        toast({
            variant: 'destructive',
            title: 'Error',
            description: 'Could not add item to cart. Please try again.',
        });
    }
  };

  const handleAddToWishlist = async () => {
    if (!user) {
      toast({
        variant: 'destructive',
        title: 'Not Logged In',
        description: 'You must be logged in to add items to your wishlist.',
      });
      return;
    }

    const wishlistRef = collection(firestore, 'users', user.uid, 'wishlist');
    
    // Check if the product is already in the wishlist
    const q = query(wishlistRef, where("productId", "==", product.id));
    const querySnapshot = await getDocs(q);

    if (!querySnapshot.empty) {
        // Product is already in the wishlist, so remove it
        const docId = querySnapshot.docs[0].id;
        await deleteDoc(doc(wishlistRef, docId));
        toast({
            title: 'Removed from Wishlist',
            description: `${product.name} has been removed from your wishlist.`,
        });
    } else {
        // Product is not in the wishlist, so add it
        const wishlistItem = {
            userId: user.uid,
            productId: product.id,
            addedAt: serverTimestamp(),
        };
        await addDoc(wishlistRef, wishlistItem);
        toast({
            title: 'Added to Wishlist',
            description: `${product.name} has been added to your wishlist.`,
        });
    }
  };
  
  const discount = product.originalPrice ? Math.round(((product.originalPrice - product.price) / product.originalPrice) * 100) : 0;

  return (
    <motion.div whileHover={{ y: -5 }} className="h-full">
      <Card className="overflow-hidden group h-full flex flex-col w-full">
        <CardHeader className="p-0 relative">
          <div className="aspect-square w-full relative">
            {image && (
              <Image
                src={image.imageUrl}
                alt={product.name}
                fill
                className="object-cover transition-transform duration-300 group-hover:scale-105"
                data-ai-hint={image.imageHint}
              />
            )}
             {discount > 0 && <Badge variant="destructive" className="absolute top-2 left-2">{discount}% OFF</Badge>}
            <div className="absolute top-2 right-2 flex flex-col gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
              <Dialog>
                <DialogTrigger asChild>
                  <Button size="icon" variant="secondary" className="h-8 w-8 rounded-full">
                    <Eye className="h-4 w-4" />
                  </Button>
                </DialogTrigger>
                <DialogContent>
                  <DialogHeader>
                    <DialogTitle>{product.name}</DialogTitle>
                  </DialogHeader>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="aspect-square w-full relative">
                        {image && <Image src={image.imageUrl} alt={product.name} fill className="object-cover rounded-md" />}
                    </div>
                    <div>
                        <p className="text-2xl font-bold">₹{product.price.toFixed(2)}</p>
                        <p className="text-sm text-muted-foreground mt-2">A detailed description of {product.name} would go here, providing users with more information about its features, benefits, and usage instructions.</p>
                        <Button className="mt-4 w-full" onClick={handleAddToCart}><ShoppingCart className="mr-2 h-4 w-4" /> Add to Cart</Button>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>
              <Button size="icon" variant="secondary" className="h-8 w-8 rounded-full" onClick={handleAddToWishlist}>
                <Heart className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent className="p-4 flex-grow">
          <p className="text-sm text-muted-foreground">{product.category}</p>
          <h3 className="font-semibold leading-tight mt-1 group-hover:text-primary transition-colors">{product.name}</h3>
           <div className="flex items-center gap-1 mt-2">
            <Star className="w-4 h-4 text-yellow-400 fill-yellow-400" />
            <span className="text-sm font-semibold">4.5</span>
            <span className="text-xs text-muted-foreground">(1,234)</span>
          </div>
        </CardContent>
        <CardFooter className="p-4 pt-0 flex justify-between items-center">
            <div>
              <p className="text-lg font-bold text-primary">₹{product.price.toFixed(2)}</p>
              {product.originalPrice && <p className="text-sm text-muted-foreground line-through">₹{product.originalPrice.toFixed(2)}</p>}
            </div>
          <Button size="sm" onClick={handleAddToCart}>
            <ShoppingCart className="mr-2 h-4 w-4" />
            Add
          </Button>
        </CardFooter>
      </Card>
    </motion.div>
  );
}
