'use client';

import React, { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { useToast } from '@/hooks/use-toast';
import { useFirestore, useUser } from '@/firebase';
import { LoaderCircle } from 'lucide-react';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from './ui/form';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Textarea } from './ui/textarea';
import { mockOrders } from '@/lib/data';
import { addDoc, collection, serverTimestamp } from 'firebase/firestore';
import { errorEmitter } from '@/firebase/error-emitter';
import { FirestorePermissionError } from '@/firebase/errors';


const ticketSchema = z.object({
  issueCategory: z.string().min(1, 'Please select a category.'),
  orderId: z.string().optional(),
  subject: z.string().min(5, 'Subject must be at least 5 characters.'),
  description: z.string().min(20, 'Description must be at least 20 characters.'),
});

type TicketFormValues = z.infer<typeof ticketSchema>;

const issueCategories = [
  'Order Issues',
  'Payment & Refunds',
  'Delivery Problems',
  'Product Information',
  'Account Issues',
  'Website Feedback',
  'Other',
];

interface SupportTicketFormProps {
  onSave?: () => void;
}

export function SupportTicketForm({ onSave }: SupportTicketFormProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();
  const { user } = useUser();
  const firestore = useFirestore();
  
  const userOrders = mockOrders;

  const form = useForm<TicketFormValues>({
    resolver: zodResolver(ticketSchema),
    defaultValues: {
      issueCategory: '',
      orderId: '',
      subject: '',
      description: '',
    },
  });

  const onSubmit = async (values: TicketFormValues) => {
    if (!user) {
      toast({
        variant: 'destructive',
        title: 'Not Authenticated',
        description: 'You must be logged in to create a ticket.',
      });
      return;
    }

    setIsSubmitting(true);
    
    const ticketData = {
        ...values,
        userId: user.uid,
        status: 'Open',
        createdAt: serverTimestamp(),
        updatedAt: serverTimestamp(),
    };

    try {
        const ticketsCollectionRef = collection(firestore, 'users', user.uid, 'tickets');
        const docRef = await addDoc(ticketsCollectionRef, ticketData);
        
        toast({
          title: 'Support Ticket Created',
          description: `Your ticket ID is ${docRef.id}. We have sent a confirmation to your email.`,
        });

        onSave?.();
        form.reset();

    } catch (error) {
        console.error("Error creating ticket:", error);
         errorEmitter.emit(
        'permission-error',
        new FirestorePermissionError({
          path: `users/${user.uid}/tickets`,
          operation: 'create',
          requestResourceData: ticketData,
        })
      );
    } finally {
        setIsSubmitting(false);
    }

  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="issueCategory"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Issue Category</FormLabel>
              <Select onValueChange={field.onChange} defaultValue={field.value}>
                <FormControl>
                  <SelectTrigger>
                    <SelectValue placeholder="Select a category" />
                  </SelectTrigger>
                </FormControl>
                <SelectContent>
                  {issueCategories.map(category => (
                    <SelectItem key={category} value={category}>{category}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <FormMessage />
            </FormItem>
          )}
        />
        
        {userOrders.length > 0 && (
          <FormField
            control={form.control}
            name="orderId"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Related Order (Optional)</FormLabel>
                <Select onValueChange={field.onChange} defaultValue={field.value}>
                  <FormControl>
                    <SelectTrigger>
                      <SelectValue placeholder="Select an order if applicable" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    {userOrders.map(order => (
                      <SelectItem key={order.id} value={order.id}>
                        #{order.id} - {order.product.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />
        )}

        <FormField
          control={form.control}
          name="subject"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Subject</FormLabel>
              <FormControl>
                <Input placeholder="e.g., 'Late Delivery'" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="description"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Problem Description</FormLabel>
              <FormControl>
                <Textarea
                  placeholder="Please describe the issue in detail."
                  className="min-h-[120px]"
                  {...field}
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        
        
        <div className="flex justify-end gap-2 pt-4">
          <Button type="submit" disabled={isSubmitting} className="w-full">
            {isSubmitting && (
              <LoaderCircle className="animate-spin mr-2 h-4 w-4" />
            )}
            Submit Ticket
          </Button>
        </div>
      </form>
    </Form>
  );
}
