
'use client';

import { useEffect } from 'react';
import { useDoc, useFirestore, useMemoFirebase } from '@/firebase';
import { doc } from 'firebase/firestore';
import type { Settings } from '@/lib/data';

function hexToHsl(hex: string): string {
  if (!hex) return '';
  hex = hex.replace('#', '');
  const r = parseInt(hex.substring(0, 2), 16) / 255;
  const g = parseInt(hex.substring(2, 4), 16) / 255;
  const b = parseInt(hex.substring(4, 6), 16) / 255;
  const max = Math.max(r, g, b);
  const min = Math.min(r, g, b);
  let h = 0, s = 0, l = (max + min) / 2;
  if (max !== min) {
    const d = max - min;
    s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
    switch (max) {
      case r: h = (g - b) / d + (g < b ? 6 : 0); break;
      case g: h = (b - r) / d + 2; break;
      case b: h = (r - g) / d + 4; break;
    }
    h /= 6;
  }
  return `${Math.round(h * 360)} ${Math.round(s * 100)}% ${Math.round(l * 100)}%`;
}

export function ThemeProvider({ children }: { children: React.ReactNode }) {
  const firestore = useFirestore();
  const settingsDocRef = useMemoFirebase(
    () => doc(firestore, 'settings', 'main'),
    [firestore]
  );
  const { data: settings } = useDoc<Settings>(settingsDocRef);

  useEffect(() => {
    if (settings) {
      const root = document.documentElement;
      const applyColor = (name: string, value?: string) => {
        if (value) {
          root.style.setProperty(`--${name}`, hexToHsl(value));
        }
      };

      applyColor('background', settings.background);
      applyColor('foreground', settings.foreground);
      applyColor('card', settings.card);
      applyColor('primary', settings.primary);
      applyColor('primary-foreground', settings.primaryForeground);
      applyColor('secondary', settings.secondary);
      applyColor('secondary-foreground', settings.secondaryForeground);
      applyColor('destructive', settings.destructive);
      applyColor('destructive-foreground', settings.destructiveForeground);
      applyColor('ring', settings.ring);
    }
  }, [settings]);

  return <>{children}</>;
}
