'use client';

import {
  doc,
  collection,
  addDoc,
  setDoc,
  deleteDoc,
  Firestore,
} from 'firebase/firestore';
import { errorEmitter } from '@/firebase/error-emitter';
import {
  FirestorePermissionError,
  type SecurityRuleContext,
} from '@/firebase/errors';
import type { Address } from '@/lib/data';

// Note: Address has an `id` property which is managed by Firestore,
// so we omit it when creating or updating data.
type AddressData = Omit<Address, 'id'>;

/**
 * Adds a new address to a user's address subcollection.
 */
export function addAddress(
  db: Firestore,
  userId: string,
  data: AddressData
) {
  const collectionRef = collection(db, 'users', userId, 'addresses');
  addDoc(collectionRef, data).catch((serverError) => {
    const permissionError = new FirestorePermissionError({
      path: collectionRef.path,
      operation: 'create',
      requestResourceData: data,
    } satisfies SecurityRuleContext);
    errorEmitter.emit('permission-error', permissionError);
  });
}

/**
 * Updates an existing address in a user's address subcollection.
 */
export function updateAddress(
  db: Firestore,
  userId: string,
  addressId: string,
  data: Partial<AddressData>
) {
  const docRef = doc(db, 'users', userId, 'addresses', addressId);
  setDoc(docRef, data, { merge: true }).catch((serverError) => {
    const permissionError = new FirestorePermissionError({
      path: docRef.path,
      operation: 'update',
      requestResourceData: data,
    } satisfies SecurityRuleContext);
    errorEmitter.emit('permission-error', permissionError);
  });
}

/**
 * Deletes an address from a user's address subcollection.
 */
export function deleteAddress(
  db: Firestore,
  userId: string,
  addressId: string
) {
  const docRef = doc(db, 'users', userId, 'addresses', addressId);
  deleteDoc(docRef).catch((serverError) => {
    const permissionError = new FirestorePermissionError({
      path: docRef.path,
      operation: 'delete',
    } satisfies SecurityRuleContext);
    errorEmitter.emit('permission-error', permissionError);
  });
}
