'use client';

import {
  doc,
  collection,
  addDoc,
  setDoc,
  deleteDoc,
  Firestore,
  serverTimestamp,
} from 'firebase/firestore';
import { errorEmitter } from '@/firebase/error-emitter';
import {
  FirestorePermissionError,
  type SecurityRuleContext,
} from '@/firebase/errors';
import type { CategoryData } from '@/lib/data';

/**
 * Adds a new category to the categories collection.
 */
export function addCategory(
  db: Firestore,
  data: Omit<CategoryData, 'id'>
) {
  const collectionRef = collection(db, 'categories');
  const categoryData = {
    ...data,
    createdAt: serverTimestamp(),
  };

  return addDoc(collectionRef, categoryData).catch((serverError) => {
    const permissionError = new FirestorePermissionError({
      path: collectionRef.path,
      operation: 'create',
      requestResourceData: categoryData,
    } satisfies SecurityRuleContext);
    errorEmitter.emit('permission-error', permissionError);
    // Do not re-throw the error here to allow the global handler to process it.
  });
}

/**
 * Updates an existing category in the categories collection.
 */
export function updateCategory(
  db: Firestore,
  categoryId: string,
  data: Partial<Omit<CategoryData, 'id'>>
) {
  const docRef = doc(db, 'categories', categoryId);
  return setDoc(docRef, data, { merge: true }).catch((serverError) => {
    const permissionError = new FirestorePermissionError({
      path: docRef.path,
      operation: 'update',
      requestResourceData: data,
    } satisfies SecurityRuleContext);
    errorEmitter.emit('permission-error', permissionError);
    // Do not re-throw the error here.
  });
}

/**
 * Deletes a category from the categories collection.
 */
export function deleteCategory(
  db: Firestore,
  categoryId: string
) {
  const docRef = doc(db, 'categories', categoryId);
  return deleteDoc(docRef).catch((serverError) => {
    const permissionError = new FirestorePermissionError({
      path: docRef.path,
      operation: 'delete',
    } satisfies SecurityRuleContext);
    errorEmitter.emit('permission-error', permissionError);
    // Do not re-throw the error here.
  });
}
