import type { LucideIcon } from 'lucide-react';
import {
  Baby,
  PersonStanding,
  Pill,
  Syringe,
  Leaf,
  Stethoscope,
  HeartPulse,
  ShoppingBasket,
} from 'lucide-react';

// This is now a representation of initial/mock data.
// The primary source of truth for categories will be Firestore.
export const initialCategories: CategoryData[] = [
  { name: 'Baby Care', icon: 'Baby' },
  { name: "Men's Care", icon: 'PersonStanding' },
  { name: "Women's Care", icon: 'HeartPulse' },
  { name: 'Medicines', icon: 'Pill' },
  { name: 'FMCG', icon: 'ShoppingBasket' },
  { name: 'Injections', icon: 'Syringe' },
  { name: 'Supplements', icon: 'Leaf' },
  { name: 'Equipment', icon: 'Stethoscope' },
];

// This type represents the data structure in Firestore
export type CategoryData = {
  id?: string; // Firestore ID
  name: string;
  icon: string; // Storing the name of the Lucide icon
  bannerUrl?: string;
};

// This type is for component props, adding the dynamically imported icon
export type Category = {
  id: string;
  name: string;
  icon: LucideIcon;
  bannerUrl?: string;
};

export type Product = {
  id: string;
  name: string;
  price: number;
  originalPrice?: number;
  imageId: string;
  category: string;
};

export type Address = {
    id: string;
    userId: string;
    fullName: string;
    mobileNumber: string;
    houseNoStreet: string;
    city: string;
    state: string;
    pincode: string;
    addressType: 'Home' | 'Work';
}

export type SupportTicketStatus =
  | 'Open'
  | 'In Progress'
  | 'Resolved'
  | 'Closed';

export type TicketMessage = {
  sender: 'user' | 'admin';
  message: string;
  timestamp: any;
};

export type SupportTicket = {
  id: string;
  userId: string;
  issueCategory: string;
  orderId?: string;
  subject: string;
  description: string;
  status: SupportTicketStatus;
  createdAt: any;
  updatedAt: any;
  conversation?: TicketMessage[];
};

export type WishlistItem = {
  id: string;
  userId: string;
  productId: string;
  addedAt: any;
  product?: Product;
}

export type CartItem = {
  id: string;
  userId: string;
  productId: string;
  quantity: number;
  addedAt: any;
};

export type User = {
    id: string;
    name: string;
    email: string;
    photoURL?: string;
    registeredDate: any;
    suspended?: boolean;
};

export type Admin = {
  id: string;
  email: string;
}

export type ReviewStatus = 'pending' | 'approved' | 'rejected';

export type Review = {
    id: string;
    userId: string;
    productId: string;
    rating: number;
    comment: string;
    status: ReviewStatus;
    createdAt: any;
};

export type NewsletterEmail = {
  id: string;
  email: string;
};

export type Coupon = {
  id?: string;
  code: string;
  discountPercentage: number;
  minOrderAmount: number;
  expiryDate: any;
  maxUsageLimit: number;
};

export type AdminNotification = {
    id: string;
    text: string;
    icon: string;
    timestamp: any;
}

export type AdminLoginLog = {
    id: string;
    adminId: string;
    email: string;
    timestamp: any;
    ipAddress: string;
    status: 'success' | 'failure';
    userAgent: string;
}

export type HeroContent = {
    id?: string;
    headline: string;
    subheading: string;
    cta1: string;
    cta2: string;
    imageId: string;
    imageUrl?: string;
    order: number;
    isActive: boolean;
};

export type HomepageSection = {
    id: string;
    name: string;
    enabled: boolean;
};

export type Settings = {
  id?: string;
  websiteName: string;
  websiteTagline?: string;
  logoUrl?: string;
  logoIcon?: string;
  
  // Theme Colors
  background?: string;
  foreground?: string;
  card?: string;
  primary?: string;
  primaryForeground?: string;
  secondary?: string;
  secondaryForeground?: string;
  destructive?: string;
  destructiveForeground?: string;
  ring?: string;

  // Payment
  stripeApiKey?: string;
  paypalClientId?: string;
  razorpayApiKey?: string;
  enableStripe?: boolean;
  enablePaypal?: boolean;
  enableRazorpay?: boolean;
  defaultCurrency?: string;
  paymentTestMode?: boolean;

  // API & Security
  publicApiKey?: string;
  ipWhitelist?: string;

  // Integrations
  googleAnalyticsId?: string;
  facebookPixelId?: string;
  googleTagManagerId?: string;

  // Performance
  enableCache?: boolean;
  imageCompression?: 'low' | 'medium' | 'high';
  
  // Maintenance
  maintenanceMode?: boolean;
  maintenancePageBanner?: string;

  // Homepage Layout
  homepageSections?: HomepageSection[];

  // Custom Code
  headerScripts?: string;
  bodyStartScripts?: string;
  footerScripts?: string;

  // SEO
  metaTitle?: string;
  metaDescription?: string;
  ogImageUrl?: string;
  websiteKeywords?: string;

  // Policies
  privacyPolicy?: string;
  termsAndConditions?: string;
  returnPolicy?: string;
};


const allProducts: Product[] = [
  {
    id: '1',
    name: 'Gentle Baby Lotion',
    price: 12.99,
    originalPrice: 15.99,
    imageId: 'prod-1',
    category: 'Baby Care',
  },
  {
    id: '2',
    name: "Men's Revitalizing Face Wash",
    price: 8.5,
    imageId: 'prod-2',
    category: "Men's Care",
  },
  {
    id: '3',
    name: 'Vitamin C Plus Zinc',
    price: 19.99,
    originalPrice: 24.99,
    imageId: 'prod-3',
    category: 'Supplements',
  },
  {
    id: '4',
    name: 'Paracetamol 500mg Tablets',
    price: 2.99,
    imageId: 'prod-4',
    category: 'Medicines',
  },
  {
    id: '5',
    name: 'Precision Digital Thermometer',
    price: 25.0,
    imageId: 'prod-5',
    category: 'Equipment',
  },
  {
    id: '6',
    name: 'Ultra-thin Sanitary Pads',
    price: 5.49,
    originalPrice: 6.99,
    imageId: 'prod-6',
    category: "Women's Care",
  },
  {
    id: '7',
    name: 'Whey Protein Isolate',
    price: 45.99,
    imageId: 'prod-7',
    category: 'Supplements',
  },
  {
    id: '8',
    name: 'Prefilled Insulin Syringe',
    price: 32.75,
    originalPrice: 35.00,
    imageId: 'prod-8',
    category: 'Injections',
  },
];

export const trendingProducts: Product[] = allProducts.slice(0, 4);
export const bestSellers: Product[] = allProducts.slice(4, 8).reverse();
export const newArrivals: Product[] = [
  ...allProducts.slice(2, 6),
  ...allProducts.slice(0, 2),
];

export type OrderStatus =
  | 'Order Placed'
  | 'Packed'
  | 'Shipped'
  | 'Out for Delivery'
  | 'Delivered';

export type Order = {
  id: string;
  orderDate: string;
  expectedDelivery: string;
  status: OrderStatus;
  product: Product;
  courier?: string;
  trackingId?: string;
};

export const mockOrders: Order[] = [
  {
    id: 'MED456-1',
    orderDate: '2024-07-28',
    expectedDelivery: '2024-08-02',
    status: 'Shipped',
    product: allProducts[3], // Paracetamol
    courier: 'BlueDart',
    trackingId: 'BD54321098',
  },
  {
    id: 'MED789-2',
    orderDate: '2024-07-29',
    expectedDelivery: '2024-08-05',
    status: 'Order Placed',
    product: allProducts[6], // Whey Protein
  },
  {
    id: 'MED123-3',
    orderDate: '2024-07-25',
    expectedDelivery: '2024-07-30',
    status: 'Delivered',
    product: allProducts[0], // Baby Lotion
    courier: 'Delhivery',
    trackingId: 'DL12345678',
  },
];

export const mockSupportTickets: Omit<SupportTicket, 'userId' | 'createdAt' | 'updatedAt' | 'description'>[] = [
  {
    id: 'TKT-123456',
    issueCategory: 'Delivery Problems',
    orderId: 'MED456-1',
    subject: 'Order not received yet',
    status: 'In Progress',
    conversation: [
        { sender: 'user', message: 'My order was supposed to be delivered yesterday, but I have not received it yet. The tracking shows it is still in transit.', timestamp: '2024-08-03T10:00:00Z' },
        { sender: 'admin', message: 'We are sorry for the delay. We are checking with the courier partner and will get back to you shortly.', timestamp: '2024-08-03T11:30:00Z' },
    ]
  },
  {
    id: 'TKT-789012',
    issueCategory: 'Product Information',
    subject: 'Question about Vitamin C',
    status: 'Resolved',
    conversation: [
        { sender: 'user', message: 'Is the Vitamin C supplement chewable?', timestamp: '2024-08-01T14:00:00Z' },
        { sender: 'admin', message: 'Yes, the Vitamin C tablets are chewable and come in a tasty orange flavor.', timestamp: '2024-08-01T14:15:00Z' },
        { sender: 'user', message: 'Great, thank you!', timestamp: '2024-08-01T14:16:00Z' },
    ]
  },
];
    
